/*
 *  This file is part of Dune Legacy.
 *
 *  Dune Legacy is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Dune Legacy is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Dune Legacy.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <players/QuantBot.h>

#include <Game.h>
#include <GameInitSettings.h>
#include <Map.h>
#include <sand.h>
#include <House.h>

#include <structures/StructureBase.h>
#include <structures/BuilderBase.h>
#include <structures/StarPort.h>
#include <structures/ConstructionYard.h>
#include <structures/RepairYard.h>
#include <structures/Palace.h>
#include <units/UnitBase.h>
#include <units/GroundUnit.h>
#include <units/AirUnit.h>
#include <units/MCV.h>
#include <units/Harvester.h>
#include <units/Saboteur.h>
#include <units/Devastator.h>
#include <units/Carryall.h>

#include <algorithm>

#define AIUPDATEINTERVAL 50



 /**
  TODO
  An old list from 2016 - Stefan van der Wel
  - fix game performance when toomany units
  ia) build concrete when no placement locations are available == in progress, bugs exist ==
  Refinerys near spice == tried but failed ==
  4. Repair yards factories, & Turrets near enemy == 50% done ==
  5. All buildings away from enemy other that silos and turrets
  i) stop repair when just on yellow (at 50%) == 50% done, still broken for some buildings as goes into yellow health ==
  ii) silo build broken == fixed ==
   carryalls sit over units hovering bug introduced.... fix scramble units and defend + manual carryall = 50% =
  4) theres a bug in on increment and decrement units...
  1. Harvesters deploy away from enemy
  5. fix gun turret & gun for rocket turret
  x. Improve squad management
  2. Make carryalls and ornithopers easier to hit
  **/



QuantBot::QuantBot(House* associatedHouse, const std::string& playername, Difficulty difficulty)
	: Player(associatedHouse, playername), difficulty(difficulty) {

	buildTimer = getRandomGen().rand(0, 3) * 50;

    attackTimer = MILLI2CYCLES(150000);
    retreatTimer = MILLI2CYCLES(60000); //turning off

	// Different AI logic based on game type
	// Campaign = Campaign AI, Skirmish/CustomGame = Custom AI
	// LoadSavegame/LoadMultiplayer preserve the original game type from save file
	// However, if we're named "AI Helper", always act as Custom AI
	
	// Log game type and AI selection details
	logDebug("QuantBot %s: Game type: %d, Player name: '%s'", 
		getPlayername().c_str(), (int)currentGame->gameType, getPlayername().c_str());

	if (getPlayername() == "AI Helper") {
		gameMode = GameMode::Custom;  // AI Helper always uses Custom mode
		logDebug("QuantBot %s: Acting as Helper AI (Custom mode)", getPlayername().c_str());
	} else if (currentGame->gameType == GameType::Campaign 
				|| currentGame->gameType == GameType::Skirmish) {
		gameMode = GameMode::Campaign;  // Campaign uses Campaign AI
		logDebug("QuantBot %s: Acting as Campaign AI (game type: %d)", getPlayername().c_str(), (int)currentGame->gameType);
	} else {
		gameMode = GameMode::Custom;  // Skirmish/CustomGame/LoadSavegame use Custom AI
		logDebug("QuantBot %s: Acting as Custom AI (game type: %d)", getPlayername().c_str(), (int)currentGame->gameType);
	} 


	if (gameMode == GameMode::Campaign) {
		// Wait a while if it is a campaign game

		switch (currentGame->techLevel) {
		case 6: {
			attackTimer = MILLI2CYCLES(540000);
		}break;

		case 7: {
			attackTimer = MILLI2CYCLES(600000);
		}break;

		case 8: {
			attackTimer = MILLI2CYCLES(720000);
		}break;

		default: {
			attackTimer = MILLI2CYCLES(480000);
		}

		}
	}
}


QuantBot::QuantBot(InputStream& stream, House* associatedHouse) : Player(stream, associatedHouse) {
	QuantBot::init();

	difficulty = static_cast<Difficulty>(stream.readUint8());
	gameMode = static_cast<GameMode>(stream.readUint8());
	buildTimer = stream.readSint32();
	attackTimer = stream.readSint32();
	retreatTimer = stream.readSint32();

	for (Uint32 i = ItemID_FirstID; i <= Structure_LastID; i++) {
		initialItemCount[i] = stream.readUint32();
	}
	initialMilitaryValue = stream.readSint32();
	militaryValueLimit = stream.readSint32();
	harvesterLimit = stream.readSint32();
	lastCalculatedSpice = stream.readSint32();
	campaignAIAttackFlag = stream.readBool();

	squadRallyLocation.x = stream.readSint32();
	squadRallyLocation.y = stream.readSint32();
	squadRetreatLocation.x = stream.readSint32();
	squadRetreatLocation.y = stream.readSint32();

	// Need to add in a building array for when people save and load
	// So that it keeps the count of buildings that should be on the map.
	Uint32 NumPlaceLocations = stream.readUint32();
	for (Uint32 i = 0; i < NumPlaceLocations; i++) {
		Sint32 x = stream.readSint32();
		Sint32 y = stream.readSint32();

		placeLocations.emplace_back(x, y);
	}
}


void QuantBot::init() {
}


QuantBot::~QuantBot() = default;

void QuantBot::save(OutputStream& stream) const {
	Player::save(stream);

	stream.writeUint8(static_cast<Uint8>(difficulty));
	stream.writeUint8(static_cast<Uint8>(gameMode));
	stream.writeSint32(buildTimer);
	stream.writeSint32(attackTimer);
	stream.writeSint32(retreatTimer);

	for (Uint32 i = ItemID_FirstID; i <= Structure_LastID; i++) {
		stream.writeUint32(initialItemCount[i]);
	}
	stream.writeSint32(initialMilitaryValue);
	stream.writeSint32(militaryValueLimit);
	stream.writeSint32(harvesterLimit);
	stream.writeSint32(lastCalculatedSpice);
	stream.writeBool(campaignAIAttackFlag);

	stream.writeSint32(squadRallyLocation.x);
	stream.writeSint32(squadRallyLocation.y);
	stream.writeSint32(squadRetreatLocation.x);
	stream.writeSint32(squadRetreatLocation.y);

	stream.writeUint32(placeLocations.size());
	for (const Coord& placeLocation : placeLocations) {
		stream.writeSint32(placeLocation.x);
		stream.writeSint32(placeLocation.y);
	}

}


void QuantBot::update() {
	// Safety check: if our house is null (e.g., during game cleanup), don't update
	if (getHouse() == nullptr) {
		return;
	}
	
	if (getGameCycleCount() == 0) {
		// The game just started and we gather some
		// Count the items once initially

		// First count all the objects we have
		for (int i = ItemID_FirstID; i <= ItemID_LastID; i++) {
			initialItemCount[i] = getHouse()->getNumItems(i);
			logDebug("Initial: Item: %d  Count: %d", i, initialItemCount[i]);
		}

		if ((initialItemCount[Structure_RepairYard] == 0) && gameMode == GameMode::Campaign && currentGame && currentGame->techLevel > 4) {
			initialItemCount[Structure_RepairYard] = 1;
			if (initialItemCount[Structure_Radar] == 0) {
				initialItemCount[Structure_Radar] = 1;
			}

			if (initialItemCount[Structure_LightFactory] == 0) {
				initialItemCount[Structure_LightFactory] = 1;
			}

			logDebug("Allow Campaign AI one Repair Yard");
		}

		// Calculate the total military value of the player
		initialMilitaryValue = 0;
		for (Uint32 i = Unit_FirstID; i <= Unit_LastID; i++) {
			if (i != Unit_Carryall
				&& i != Unit_Harvester
				&& i != Unit_MCV
				&& i != Unit_Sandworm) {
				// Used for campaign mode.
				if (currentGame) {
					initialMilitaryValue += initialItemCount[i] * currentGame->objectData.data[i][getHouse()->getHouseID()].price;
				}
			}
		}



		switch (gameMode) {
		case GameMode::Campaign: {

			switch (difficulty) {
			case Difficulty::Easy: {
				harvesterLimit = initialItemCount[Structure_Refinery];
				if (currentGame && currentGame->getGameInitSettings().getMission() >= 21
					&& initialMilitaryValue < 2000) {
					militaryValueLimit = 2000;
				}
				else {
					militaryValueLimit = initialMilitaryValue;
				}

				logDebug("Easy Campaign - harvesterLimit: %d", harvesterLimit);
			} break;

			case Difficulty::Medium: {
				harvesterLimit = 2 * initialItemCount[Structure_Refinery];
				militaryValueLimit = lround(initialMilitaryValue * 1.5_fix);
				if (militaryValueLimit < 4000 && currentGame && currentGame->getGameInitSettings().getMission() >= 21) {
					militaryValueLimit = 4000;
				}

				logDebug("Medium Campaign - harvesterLimit: %d", harvesterLimit);
			} break;

			case Difficulty::Hard: {
				if (currentGame && currentGame->getGameInitSettings().getMission() >= 21) {
					initialItemCount[Structure_Refinery] = 2;
					militaryValueLimit = 10000;
				}
				else {
					
					militaryValueLimit = lround(initialMilitaryValue * 2_fix);
				}
				harvesterLimit = 2 * initialItemCount[Structure_Refinery];

				logDebug("Hard Campaign - harvesterLimit: %d", harvesterLimit);
			} break;

			case Difficulty::Brutal: {
				//harvesterLimit = (currentGameMap->getSizeX() * currentGameMap->getSizeY() / 512);
				if (initialItemCount[Structure_Refinery] < 2) {
					initialItemCount[Structure_Refinery] = 2;
				}
				militaryValueLimit = lround(initialMilitaryValue * 3_fix);
				harvesterLimit = 3 * initialItemCount[Structure_Refinery];
				logDebug("Brutal Campaign - harvesterLimit: %d", harvesterLimit);
			} break;

			case Difficulty::Defend: {
				harvesterLimit = 2 * initialItemCount[Structure_Refinery];
				militaryValueLimit = lround(initialMilitaryValue * 1.8_fix);

				logDebug("Defensive Campaign - harvesterLimit: %d", harvesterLimit);
			} break;
			}

		} break;

		case GameMode::Custom: {
			// set initial unit position
			findSquadRallyLocation();
			retreatAllUnits();

			// Set harvester limits based on map size and difficulty
			int mapsize = 4096; // Default fallback size
			if (currentGameMap) {
				mapsize = currentGameMap->getSizeX() * currentGameMap->getSizeY();
			}
			
			switch (difficulty) {
			case Difficulty::Easy: {
				if (mapsize <= 1024) {
					harvesterLimit = 2;  // 32x32
					militaryValueLimit = 3000;  // 32x32
				} else if (mapsize <= 4096) {
					harvesterLimit = 2;  // 62x62, 64x64
					militaryValueLimit = 8000;  // 64x64
				} else if (mapsize <= 16384) {
					harvesterLimit = 8;  // 128x128
					militaryValueLimit = 20000;  // 128x128
				} else {
					harvesterLimit = 8 * (mapsize / 16384.0);  // Scale for larger maps
					militaryValueLimit = 20000 * (mapsize / 16384.0);
				}
				logDebug("BUILD EASY CUSTOM. mapsize: %d, harvesterLimit: %d, militaryValueLimit: %d", mapsize, harvesterLimit, militaryValueLimit);
			} break;

			case Difficulty::Medium: {
				if (mapsize <= 1024) {
					harvesterLimit = 3;  // 32x32
					militaryValueLimit = 5000;  // 32x32
				} else if (mapsize <= 4096) {
					harvesterLimit = 4;  // 62x62, 64x64
					militaryValueLimit = 12000;  // 64x64
				} else if (mapsize <= 16384) {
					harvesterLimit = 15;  // 128x128
					militaryValueLimit = 35000;  // 128x128
				} else {
					harvesterLimit = 15 * (mapsize / 16384.0);  // Scale for larger maps
					militaryValueLimit = 35000 * (mapsize / 16384.0);
				}
				logDebug("BUILD MEDIUM CUSTOM. mapsize: %d, harvesterLimit: %d, militaryValueLimit: %d", mapsize, harvesterLimit, militaryValueLimit);
			} break;

			case Difficulty::Hard: {
				if (mapsize <= 1024) {
					harvesterLimit = 4;  // 32x32
					militaryValueLimit = 8000;  // 32x32
				} else if (mapsize <= 4096) {
					harvesterLimit = 7;  // 62x62, 64x64
					militaryValueLimit = 20000;  // 64x64
				} else if (mapsize <= 16384) {
					harvesterLimit = 40;  // 128x128
					militaryValueLimit = 50000;  // 128x128
				} else {
					harvesterLimit = 40 * (mapsize / 16384.0);  // Scale for larger maps
					militaryValueLimit = 50000 * (mapsize / 16384.0);
				}
				logDebug("BUILD HARD CUSTOM. mapsize: %d, harvesterLimit: %d, militaryValueLimit: %d", mapsize, harvesterLimit, militaryValueLimit);
			} break;

			case Difficulty::Brutal: {
				if (mapsize <= 1024) {
					harvesterLimit = 10;  // 32x32 - increased from 5
					militaryValueLimit = 20000;  // 32x32 - increased from 12000
				} else if (mapsize <= 4096) {
					harvesterLimit = 20;  // 62x62, 64x64 - increased from 10
					militaryValueLimit = 40000;  // 64x64 - increased from 30000
				} else if (mapsize <= 16384) {
					harvesterLimit = 100;  // 128x128 - increased from 70 to 100
					militaryValueLimit = 80000;  // 128x128 - increased from 75000 to 80000
				} else {
					harvesterLimit = 100 * (mapsize / 16384.0);  // Scale for larger maps - base increased to 100
					militaryValueLimit = 80000 * (mapsize / 16384.0);  // Scale for larger maps - base increased to 80000
				}
				logDebug("BUILD BRUTAL CUSTOM. mapsize: %d, harvesterLimit: %d, militaryValueLimit: %d", mapsize, harvesterLimit, militaryValueLimit);
			} break;

			case Difficulty::Defend: {
				if (mapsize <= 1024) {
					harvesterLimit = 3;  // 32x32
					militaryValueLimit = 4000;  // 32x32
				} else if (mapsize <= 4096) {
					harvesterLimit = 4;  // 62x62, 64x64
					militaryValueLimit = 10000;  // 64x64
				} else if (mapsize <= 16384) {
					harvesterLimit = 15;  // 128x128
					militaryValueLimit = 25000;  // 128x128
				} else {
					harvesterLimit = 15 * (mapsize / 16384.0);  // Scale for larger maps
					militaryValueLimit = 25000 * (mapsize / 16384.0);
				}
				logDebug("BUILD DEFEND CUSTOM. mapsize: %d, harvesterLimit: %d, militaryValueLimit: %d", mapsize, harvesterLimit, militaryValueLimit);
			} break;
			}

			// what is this useful for? Reseting limits or something
			/*
			if ((currentGameMap->getSizeX() * currentGameMap->getSizeY() / 480) < harvesterLimit && difficulty != Difficulty::Brutal) {
				harvesterLimit = currentGameMap->getSizeX() * currentGameMap->getSizeY() / 480;
				logDebug("Reset harvesterLimit: %d = mapX: %d * mapY: %d / 480", harvesterLimit, currentGameMap->getSizeX(), currentGameMap->getSizeY());
			}*/

		} break;

		}
		
		// Calculate total spice remaining on map and adjust harvester limit for both modes
		lastCalculatedSpice = 0;
		if (currentGameMap) {
			const int mapSizeX = currentGameMap->getSizeX();
			const int mapSizeY = currentGameMap->getSizeY();
			
			for (int x = 0; x < mapSizeX; x++) {
				for (int y = 0; y < mapSizeY; y++) {
					Tile* pTile = currentGameMap->getTile(x, y);
					if (pTile && pTile->hasSpice()) {
						lastCalculatedSpice += pTile->getSpice().lround();
					}
				}
			}
		}
		
		// Apply spice-based harvester limit only for Custom mode
		if (gameMode == GameMode::Custom) {
			// Don't build more harvesters if total spice < 2000 * harvester count
			int maxHarvestersForSpice = lastCalculatedSpice / 2000;
			if (maxHarvestersForSpice < harvesterLimit) {
				harvesterLimit = std::max(1, maxHarvestersForSpice); // Always allow at least 1 harvester
				logDebug("Harvester limit reduced due to low spice: %d (spice: %d)", harvesterLimit, lastCalculatedSpice);
			}
		}
		
		logDebug("Initial spice calculation: %d spice remaining on map", lastCalculatedSpice);
	}

	// Recalculate spice every AI update cycle for both Campaign and Custom modes
	// Do this BEFORE the AI update interval check so it always happens
	lastCalculatedSpice = 0;
	if (currentGameMap) {
		const int mapSizeX = currentGameMap->getSizeX();
		const int mapSizeY = currentGameMap->getSizeY();
		
		for (int x = 0; x < mapSizeX; x++) {
			for (int y = 0; y < mapSizeY; y++) {
				Tile* pTile = currentGameMap->getTile(x, y);
				if (pTile && pTile->hasSpice()) {
					lastCalculatedSpice += pTile->getSpice().lround();
				}
			}
		}
	}

	if ((getGameCycleCount() + getHouse()->getHouseID()) % AIUPDATEINTERVAL != 0) {
		// we are not updating this AI player this cycle
		return;
	}

	// Calculate the total military value of the player
	int militaryValue = 0;
	for (Uint32 i = Unit_FirstID; i <= Unit_LastID; i++) {
		if (i != Unit_Carryall
			&& i != Unit_Harvester
			&& i != Unit_MCV
			&& i != Unit_Sandworm) {
			if (currentGame) {
				militaryValue += getHouse()->getNumItems(i) * currentGame->objectData.data[i][getHouse()->getHouseID()].price;
			}
		}
	}
	//logDebug("Military Value %d  Initial Military Value %d", militaryValue, initialMilitaryValue);

	checkAllUnits();

	if (buildTimer <= 0) {
		build(militaryValue);
	}
	else {
		buildTimer -= AIUPDATEINTERVAL;
	}

	if (attackTimer <= 0) {
		attack(militaryValue);
	}

	else {
		attackTimer -= AIUPDATEINTERVAL;
	}
}


void QuantBot::onObjectWasBuilt(const ObjectBase* pObject) {
}


void QuantBot::onDecrementStructures(int itemID, const Coord& location) {
}


/// When we take losses we should hold off from attacking for longer...
void QuantBot::onDecrementUnits(int itemID) {
	// Safety check: if our house is null (e.g., during game cleanup), don't process
	if (getHouse() == nullptr) {
		return;
	}
	
	if (itemID != Unit_Trooper && itemID != Unit_Infantry) {
		//attackTimer += MILLI2CYCLES(currentGame->objectData.data[itemID][getHouse()->getHouseID()].price * 30 / (static_cast<Uint8>(difficulty) + 1));
		//logDebug("loss ");
		if (currentGame) {
			retreatTimer -= MILLI2CYCLES(currentGame->objectData.data[itemID][getHouse()->getHouseID()].price * 20);
		}
	}
}


/// When we get kills we should re-attack sooner...
void QuantBot::onIncrementUnitKills(int itemID) {
	// Safety check: if our house is null (e.g., during game cleanup), don't process
	if (getHouse() == nullptr) {
		return;
	}
	
	if (itemID != Unit_Trooper && itemID != Unit_Infantry) {
		//if (currentGame) {
		//	attackTimer -= MILLI2CYCLES(currentGame->objectData.data[itemID][getHouse()->getHouseID()].price * 15);
		//}
		//logDebug("kill ");
	}
}

void QuantBot::onDamage(const ObjectBase* pObject, int damage, Uint32 damagerID) {
	// Safety check: if our house is null (e.g., during game cleanup), don't process
	if (getHouse() == nullptr) {
		return;
	}
	
	// Safety check: if pObject is null, don't process
	if (pObject == nullptr) {
		return;
	}
	
	const ObjectBase* pDamager = getObject(damagerID);

	if (pDamager == nullptr || pDamager->getOwner() == getHouse() || pObject->getItemID() == Unit_Sandworm) {
		return;
	}

    // If the human has attacked us then its time to start fighting back... unless its an attack on a special unit
    // Don't trigger with fremen or saboteur
    bool bPossiblyOwnFremen = (pObject->getOwner()->getHouseID() == HOUSE_ATREIDES) && (pObject->getItemID() == Unit_Trooper) && (currentGame && currentGame->techLevel > 7);
    if(gameMode == GameMode::Campaign && !pDamager->getOwner()->isAI() && !campaignAIAttackFlag && !bPossiblyOwnFremen && (pObject->getItemID() != Unit_Saboteur)) {
        campaignAIAttackFlag = true;
    } else if (pObject->isAStructure()) {
        doRepair(pObject);
        // no point scrambling to defend a missile
        if(pDamager->getItemID() != Structure_Palace) {
            int numStructureDefenders = 0;
            switch(difficulty) {
                case Difficulty::Defend:    numStructureDefenders = 4;                                  break;
                case Difficulty::Easy:      numStructureDefenders = 6;                                  break;
                case Difficulty::Medium:    numStructureDefenders = 10;                                 break;
                case Difficulty::Hard:      numStructureDefenders = 20;                                 break;
                case Difficulty::Brutal:    numStructureDefenders = std::numeric_limits<int>::max();    break;
            }
            scrambleUnitsAndDefend(pDamager, numStructureDefenders);
        }

	}
	else if (pObject->isAGroundUnit()) {
		const GroundUnit* pGroundUnit = static_cast<const GroundUnit*>(pObject);

		// Only command units we own
		if (pGroundUnit->getOwner() != getHouse()) {
			return;
		}

		Coord squadCenterLocation = findSquadCenter(pGroundUnit->getOwner()->getHouseID());

		if (pGroundUnit->isAwaitingPickup()) {
			return;
		}

		// Stop him dead in his tracks if he's going to rally point
		if (pGroundUnit->wasForced() && (pGroundUnit->getItemID() != Unit_Harvester)) {
			doMove2Pos(pGroundUnit,
				pGroundUnit->getCenterPoint().x,
				pGroundUnit->getCenterPoint().y,
				false);
		}

		if (pGroundUnit->getItemID() == Unit_Harvester) {
			// Always keep Harvesters away from harm
			// Defend the harvester!
			const Harvester* pHarvester = static_cast<const Harvester*>(pGroundUnit);
			if (pHarvester->isActive() && (!pHarvester->isReturning()) && pHarvester->getAmountOfSpice() > 0) {
				int numHarvesterDefenders = 0;
				switch (difficulty) {
				case Difficulty::Defend:    numHarvesterDefenders = 2;                                  break;
				case Difficulty::Easy:      numHarvesterDefenders = 3;                                  break;
				case Difficulty::Medium:    numHarvesterDefenders = 5;                                  break;
				case Difficulty::Hard:      numHarvesterDefenders = 10;                                 break;
				case Difficulty::Brutal:    numHarvesterDefenders = std::numeric_limits<int>::max();    break;
				}
				scrambleUnitsAndDefend(pDamager, numHarvesterDefenders);
				doReturn(pHarvester);
			}
		}
		// Launcher retreat logic removed from onDamage() to prevent spam
		// It's handled in checkAllUnits() instead
		else if ((currentGame && currentGame->techLevel > 3)
			&& (pGroundUnit->getItemID() == Unit_Quad)
			&& !pDamager->isInfantry()
			&& (pDamager->getItemID() != Unit_RaiderTrike)
			&& (pDamager->getItemID() != Unit_Trike)
			&& (pDamager->getItemID() != Unit_Quad)) {
			// We want out quads as raiders
			// Quads flee from every unit except trikes, infantry and other quads (but only if quads are not our main vehicle for that techlevel)
			doSetAttackMode(pGroundUnit, AREAGUARD);
			doMove2Pos(pGroundUnit, squadCenterLocation.x, squadCenterLocation.y, true);
		}
		else if ((currentGame && currentGame->techLevel > 3)
			&& ((pGroundUnit->getItemID() == Unit_RaiderTrike) || (pGroundUnit->getItemID() == Unit_Trike))
			&& !pDamager->isInfantry()
			&& (pDamager->getItemID() != Unit_RaiderTrike)
			&& (pDamager->getItemID() != Unit_Trike)) {
			// Quads flee from every unit except infantry and other trikes (but only if trikes are not our main vehicle for that techlevel)
			// We want to use our light vehicles as raiders.
			// This means they are free to engage other light military units
			// but should run away from tanks

			doSetAttackMode(pGroundUnit, AREAGUARD);
			doMove2Pos(pGroundUnit, squadCenterLocation.x, squadCenterLocation.y, true);

		}

		// If unit is below 80% then rotate them
		// If the unit is at 70% health or less and is not being forced to move anywhere
		// only do these actions for vehicles and not when fighting turrets
		// repair them, if they are eligible to be repaired
		if (difficulty != Difficulty::Easy) {
			if (pGroundUnit->getHealth() / pGroundUnit->getMaxHealth() < 0.80_fix
				&& !pGroundUnit->isInfantry()
				&& pGroundUnit->isVisible()
				&& (pDamager->getItemID() != Structure_GunTurret
					&& pDamager->getItemID() != Structure_RocketTurret)
				) {

				// If unit isn't an infantry then heal it once it is below 70% health if not an easy or medium campaign
				if (getHouse()->hasRepairYard()
					&& pGroundUnit->getHealth() / pGroundUnit->getMaxHealth() < 0.7_fix

					// don't do manual repairs if it's campaign and easy or medium difficulty
					&& !(gameMode == GameMode::Campaign && (difficulty == Difficulty::Easy || difficulty == Difficulty::Medium))
					) {
					doRepair(pGroundUnit);
				}

				// Retreat to squad center logic removed as requested
			}
		}
	}
}

Coord QuantBot::findMcvPlaceLocation(const MCV* pMCV) {
	Coord bestLocation = findPlaceLocation(Structure_ConstructionYard);

	if (bestLocation == Coord::Invalid()) {
		logDebug("No MCV deploy location adjacent to existing base structures was found, move to full search | ");

		int bestLocationScore = 1000;

		// Allow placement on map edge
		for (int placeLocationX = 0; placeLocationX < getMap().getSizeX(); placeLocationX++) {
			for (int placeLocationY = 0; placeLocationY < getMap().getSizeY(); placeLocationY++) {
				Coord placeLocation = Coord::Invalid();
				placeLocation.x = placeLocationX;
				placeLocation.y = placeLocationY;

				if (getMap().okayToPlaceStructure(placeLocationX, placeLocationY, 2, 2, false, nullptr)) {
					int locationScore = lround(blockDistance(pMCV->getLocation(), placeLocation));
					if (locationScore < bestLocationScore) {
						bestLocationScore = locationScore;
						bestLocation.x = placeLocationX;
						bestLocation.y = placeLocationY;
					}
				}
			}
		}
	}

	return bestLocation;
}

Coord QuantBot::findPlaceLocation(Uint32 itemID) {
	// Will over allocate space for small maps so its not clean
	// But should allow Richard to compile
	int buildLocationScore[128][128] = { {0} };

	int bestLocationX = -1;
	int bestLocationY = -1;
	int bestLocationScore = -10000;
	int newSizeX = getStructureSize(itemID).x;
	int newSizeY = getStructureSize(itemID).y;
	Coord bestLocation = Coord::Invalid();

	for (const StructureBase* pStructureExisting : getStructureList()) {
		if (pStructureExisting->getOwner() == getHouse()) {

			int existingStartX = pStructureExisting->getX();
			int existingStartY = pStructureExisting->getY();

			int existingSizeX = pStructureExisting->getStructureSizeX();
			int existingSizeY = pStructureExisting->getStructureSizeY();

			int existingEndX = existingStartX + existingSizeX;
			int existingEndY = existingStartY + existingSizeY;

			squadRallyLocation = findSquadRallyLocation();

			bool existingIsBuilder = (pStructureExisting->getItemID() == Structure_HeavyFactory
				|| pStructureExisting->getItemID() == Structure_RepairYard
				|| pStructureExisting->getItemID() == Structure_LightFactory
				|| pStructureExisting->getItemID() == Structure_WOR
				|| pStructureExisting->getItemID() == Structure_Barracks
				|| pStructureExisting->getItemID() == Structure_StarPort);

			bool sizeMatchX = (existingSizeX == newSizeX);
			bool sizeMatchY = (existingSizeY == newSizeY);


			for (int placeLocationX = existingStartX - newSizeX; placeLocationX <= existingEndX; placeLocationX++) {
				for (int placeLocationY = existingStartY - newSizeY; placeLocationY <= existingEndY; placeLocationY++) {
					if (getMap().tileExists(placeLocationX, placeLocationY)) {
						if (getMap().okayToPlaceStructure(placeLocationX, placeLocationY, newSizeX, newSizeY,
							false, (itemID == Structure_ConstructionYard) ? nullptr : getHouse())) {

							int placeLocationEndX = placeLocationX + newSizeX;
							int placeLocationEndY = placeLocationY + newSizeY;

							bool alignedX = (placeLocationX == existingStartX && sizeMatchX);
							bool alignedY = (placeLocationY == existingStartY && sizeMatchY);

							// bool placeGapExists = (placeLocationEndX < existingStartX || placeLocationX > existingEndX || placeLocationEndY < existingStartY || placeLocationY > existingEndY);

							// How many free spaces the building will have if placed
							// Check 2 tiles away from buildings since you can place buildings 1 tile away from existing buildings
							for (int i = placeLocationX - 2; i <= placeLocationEndX + 1; i++) {
								for (int j = placeLocationY - 2; j <= placeLocationEndY + 1; j++) {
									if (getMap().tileExists(i, j) && (getMap().getSizeX() > i) && (0 <= i) && (getMap().getSizeY() > j) && (0 <= j)) {
										// Favor edge of map placement
										if ((i == 0) || (i == getMap().getSizeX() - 1) || (j == 0) || (j == getMap().getSizeY() - 1)) {
											buildLocationScore[placeLocationX][placeLocationY] += 10;
										}

										if (getMap().getTile(i, j)->hasAStructure()) {
											// If one of our buildings is nearby favour the location
											// if it is someone elses building don't favour it
											if (getMap().getTile(i, j)->getOwner() == getHouse()->getHouseID()) {
												buildLocationScore[placeLocationX][placeLocationY] += 3;
											}
											else {
												buildLocationScore[placeLocationX][placeLocationY] -= 10;
											}
										}
										else if (!getMap().getTile(i, j)->isRock()) {
											// square isn't rock, favour it
											buildLocationScore[placeLocationX][placeLocationY] += 5;
										}
										else if (getMap().getTile(i, j)->hasAGroundObject()) {
											if (getMap().getTile(i, j)->getOwner() != getHouse()->getHouseID()) {
												// try not to build next to units which aren't yours
												buildLocationScore[placeLocationX][placeLocationY] -= 100;
											}
											else if (itemID != Structure_RocketTurret) {
												buildLocationScore[placeLocationX][placeLocationY] -= 20;
											}
										}
									}
									else {
										// penalise if outside of map
										buildLocationScore[placeLocationX][placeLocationY] -= 200;
									}
								}
							}

							//encourage structure alignment
							if (alignedX) {
								buildLocationScore[placeLocationX][placeLocationX] += 10;
							}

							if (alignedY) {
								buildLocationScore[placeLocationX][placeLocationY] += 10;
							}

							// Add building specific scores
							if (existingIsBuilder) {
								// Builders should be closer to rally point and base center
								buildLocationScore[placeLocationX][placeLocationY] -= lround(blockDistance(squadRallyLocation, Coord(placeLocationX, placeLocationY)) / 2);
								buildLocationScore[placeLocationX][placeLocationY] -= lround(blockDistance(findBaseCentre(getHouse()->getHouseID()), Coord(placeLocationX, placeLocationY)));
							}
							else if (itemID == Structure_GunTurret || itemID == Structure_RocketTurret) {
								// Turrets should prefer map edges for defensive positioning
								// Closer to map edge = higher score
								int distanceToEdge = std::min({placeLocationX, placeLocationY, 
									getMap().getSizeX() - 1 - placeLocationX, getMap().getSizeY() - 1 - placeLocationY});
								buildLocationScore[placeLocationX][placeLocationY] += (10 - distanceToEdge) * 5; // Favor edges more
								
								// Rocket turrets should also favor being close to squad rally point for better unit protection
								if (itemID == Structure_RocketTurret) {
									FixPoint distanceToRally = blockDistance(squadRallyLocation, Coord(placeLocationX, placeLocationY));
									buildLocationScore[placeLocationX][placeLocationY] += lround(30 - distanceToRally * 2); // Bonus for being close to rally point
								}
							}
							else if (itemID == Structure_Refinery) {
								// Refineries should prefer being close to spice deposits
								FixPoint closestSpiceDistance = FixPt_MAX;
								for (int spiceX = 0; spiceX < getMap().getSizeX(); spiceX++) {
									for (int spiceY = 0; spiceY < getMap().getSizeY(); spiceY++) {
										if (getMap().tileExists(spiceX, spiceY) && getMap().getTile(spiceX, spiceY)->hasSpice()) {
											FixPoint spiceDistance = blockDistance(Coord(placeLocationX, placeLocationY), Coord(spiceX, spiceY));
											if (spiceDistance < closestSpiceDistance) {
												closestSpiceDistance = spiceDistance;
											}
										}
									}
								}
								// Higher bonus for being closer to spice (max bonus of 50 points for adjacent to spice)
								if (closestSpiceDistance < FixPt_MAX) {
									buildLocationScore[placeLocationX][placeLocationY] += lround(50 - closestSpiceDistance * 2);
								}
							}

							// Pick this location if it has the best score
							if (buildLocationScore[placeLocationX][placeLocationY] > bestLocationScore) {
								bestLocationScore = buildLocationScore[placeLocationX][placeLocationY];
								bestLocationX = placeLocationX;
								bestLocationY = placeLocationY;
								//logDebug("Build location for item:%d  x:%d y:%d score:%d", itemID, bestLocationX, bestLocationY, bestLocationScore);
							}
						}
					}
				}
			}
		}
	}


	if (bestLocationScore != -10000) {
		bestLocation = Coord(bestLocationX, bestLocationY);
	}

	return bestLocation;
}

Coord QuantBot::findPlaceLocationSimple(Uint32 itemID) {
	int newSizeX = getStructureSize(itemID).x;
	int newSizeY = getStructureSize(itemID).y;
	
	squadRallyLocation = findSquadRallyLocation();
	
	FixPoint bestScore = -FixPt_MAX;
	Coord bestLocation = Coord::Invalid();
	
	// Check every tile on the map for valid placement
	for (int x = 0; x <= getMap().getSizeX() - newSizeX; x++) {
		for (int y = 0; y <= getMap().getSizeY() - newSizeY; y++) {
			// First check if this location is valid for building
			if (getMap().okayToPlaceStructure(x, y, newSizeX, newSizeY, false, 
				(itemID == Structure_ConstructionYard) ? nullptr : getHouse())) {
				
				FixPoint score = 0;
				
				// Base scoring - favor being close to existing buildings
				FixPoint closestOwnBuildingDistance = FixPt_MAX;
				for (const StructureBase* pStructure : getStructureList()) {
					if (pStructure->getOwner() == getHouse()) {
						FixPoint distance = blockDistance(Coord(x, y), Coord(pStructure->getX(), pStructure->getY()));
						if (distance < closestOwnBuildingDistance) {
							closestOwnBuildingDistance = distance;
						}
					}
				}
				if (closestOwnBuildingDistance < FixPt_MAX) {
					score += 50 - closestOwnBuildingDistance; // Bonus for being close to our buildings
				}
				
				// Building-specific placement preferences
				if (itemID == Structure_GunTurret || itemID == Structure_RocketTurret) {
					// Turrets prefer map edges for defensive positioning
					int distanceToEdge = std::min({x, y, getMap().getSizeX() - 1 - x, getMap().getSizeY() - 1 - y});
					score += (10 - distanceToEdge) * 5; // Higher score for being closer to edges
					
					// Rocket turrets also prefer being close to squad rally point
					if (itemID == Structure_RocketTurret) {
						FixPoint distanceToRally = blockDistance(squadRallyLocation, Coord(x, y));
						score += 30 - distanceToRally * 2; // Bonus for being close to rally point
					}
				}
				else if (itemID == Structure_Refinery) {
					// Refineries prefer being close to spice deposits
					FixPoint closestSpiceDistance = FixPt_MAX;
					for (int spiceX = 0; spiceX < getMap().getSizeX(); spiceX++) {
						for (int spiceY = 0; spiceY < getMap().getSizeY(); spiceY++) {
							if (getMap().tileExists(spiceX, spiceY) && getMap().getTile(spiceX, spiceY)->hasSpice()) {
								FixPoint spiceDistance = blockDistance(Coord(x, y), Coord(spiceX, spiceY));
								if (spiceDistance < closestSpiceDistance) {
									closestSpiceDistance = spiceDistance;
								}
							}
						}
					}
					if (closestSpiceDistance < FixPt_MAX) {
						score += 50 - closestSpiceDistance * 2; // Higher bonus for being closer to spice
					}
				}
				else if (itemID == Structure_HeavyFactory || itemID == Structure_LightFactory || 
						 itemID == Structure_WOR || itemID == Structure_Barracks || itemID == Structure_StarPort) {
					// Production buildings prefer being close to rally point and base center
					FixPoint distanceToRally = blockDistance(squadRallyLocation, Coord(x, y));
					FixPoint distanceToBase = blockDistance(findBaseCentre(getHouse()->getHouseID()), Coord(x, y));
					score += 20 - distanceToRally / 2; // Bonus for being close to rally point
					score += 20 - distanceToBase; // Bonus for being close to base center
				}
				
				// Favor map edges in general for defensive positioning
				if (x == 0 || x == getMap().getSizeX() - newSizeX || y == 0 || y == getMap().getSizeY() - newSizeY) {
					score += 10;
				}
				
				// Check if this is the best location so far
				if (score > bestScore) {
					bestScore = score;
					bestLocation = Coord(x, y);
				}
			}
		}
	}
	
	return bestLocation;
}

void QuantBot::build(int militaryValue) {
	// Safety check: if our house is null (e.g., during game cleanup), don't build
	if (getHouse() == nullptr) {
		return;
	}
	
	int houseID = getHouse()->getHouseID();
	
	// Safety check: if currentGame is null, don't build
	if (currentGame == nullptr) {
		return;
	}
	
	auto& data = currentGame->objectData.data;

	int itemCount[Num_ItemID];
	for (int i = ItemID_FirstID; i <= ItemID_LastID; i++) {
		itemCount[i] = getHouse()->getNumItems(i);
	}

	int activeHeavyFactoryCount = 0;
	int activeRepairYardCount = 0;
	int activeHighTechFactoryCount = 0;

	// Let's try just running this once...
	if (squadRallyLocation.isInvalid()) {
		squadRallyLocation = findSquadRallyLocation();
		squadRetreatLocation = findSquadRetreatLocation();
		if (gameMode != GameMode::Campaign) {
			retreatAllUnits();
		}
	}

	// Move all units to squad rally point early in game to clear building areas
	// Do this for the first 2 minutes of the game to ensure units don't block construction
	if (getGameCycleCount() < MILLI2CYCLES(120000)) { // First 2 minutes
		for (const UnitBase* pUnit : getUnitList()) {
			if (pUnit->getOwner() == getHouse() 
				&& pUnit->getItemID() != Unit_Harvester // Don't move harvesters
				&& pUnit->getItemID() != Unit_MCV       // Don't move MCVs (handled separately)
				&& pUnit->getItemID() != Unit_Carryall  // Don't move carryalls
				&& pUnit->getItemID() != Unit_Frigate   // Don't move frigates
				&& pUnit->getItemID() != Unit_Sandworm  // Don't move sandworms
				&& !pUnit->hasATarget()                 // Don't interrupt combat
				&& !pUnit->wasForced()                  // Don't override player commands
				&& squadRallyLocation.isValid()) {
				
				// Move to squad rally point if not already there or heading there
				if (blockDistance(pUnit->getLocation(), squadRallyLocation) > 3 
					&& pUnit->getDestination() != squadRallyLocation) {
					doMove2Pos(const_cast<UnitBase*>(pUnit), squadRallyLocation.x, squadRallyLocation.y, false);
				}
			}
		}
	}

	// Next add in the objects we are building
	for (const StructureBase* pStructure : getStructureList()) {
		if (pStructure && pStructure->getOwner() == getHouse()) {
			if (pStructure->isABuilder()) {
				const BuilderBase* pBuilder = static_cast<const BuilderBase*>(pStructure);
				if (pBuilder->getProductionQueueSize() > 0) {
					itemCount[pBuilder->getCurrentProducedItem()]++;
					if (pBuilder->getItemID() == Structure_HeavyFactory) {
						activeHeavyFactoryCount++;
					} else if (pBuilder->getItemID() == Structure_HighTechFactory) {
						activeHighTechFactoryCount++;
					}
				}
			}
			else if (pStructure->getItemID() == Structure_RepairYard) {
				const RepairYard* pRepairYard = static_cast<const RepairYard*>(pStructure);
				if (!pRepairYard->isFree()) {
					activeRepairYardCount++;
				}

			}

			// Set unit deployment position
			if (pStructure->getItemID() == Structure_Barracks
				|| pStructure->getItemID() == Structure_WOR
				|| pStructure->getItemID() == Structure_LightFactory
				|| pStructure->getItemID() == Structure_HeavyFactory
				|| pStructure->getItemID() == Structure_RepairYard
				|| pStructure->getItemID() == Structure_StarPort) {
				doSetDeployPosition(pStructure, squadRallyLocation.x, squadRallyLocation.y);
			}
		}


	}

	int money = getHouse()->getCredits();

	if (militaryValue > 0 || getHouse()->getNumStructures() > 0) {
		if (gameMode == GameMode::Custom) {
			logDebug("Stats: %d  crdt: %d  mVal: %d/%d  built: %d  kill: %d  loss: %d remaining spice: %d hvstr: %d/%d",
				attackTimer, getHouse()->getCredits(), militaryValue, militaryValueLimit, getHouse()->getUnitBuiltValue(),
				getHouse()->getKillValue(), getHouse()->getLossValue(), lastCalculatedSpice, getHouse()->getNumItems(Unit_Harvester), harvesterLimit);
		} else {
			logDebug("Stats: %d  crdt: %d  mVal: %d/%d  built: %d  kill: %d  loss: %d hvstr: %d/%d",
				attackTimer, getHouse()->getCredits(), militaryValue, militaryValueLimit, getHouse()->getUnitBuiltValue(),
				getHouse()->getKillValue(), getHouse()->getLossValue(), getHouse()->getNumItems(Unit_Harvester), harvesterLimit);
		}
	}


	// Second attempt at unit prioritisation
	// This algorithm calculates damage dealt over units lost value for each unit type
	// referred to as damage loss ratio (dlr)
	// It then prioritises the build of units with a higher dlr



	FixPoint dlrTank = getHouse()->getNumItemDamageInflicted(Unit_Tank) / FixPoint((1 + getHouse()->getNumLostItems(Unit_Tank)) * data[Unit_Tank][houseID].price);
	FixPoint dlrSiege = getHouse()->getNumItemDamageInflicted(Unit_SiegeTank) / FixPoint((1 + getHouse()->getNumLostItems(Unit_SiegeTank)) * data[Unit_SiegeTank][houseID].price);
	int numSpecialUnitsDamageInflicted = getHouse()->getNumItemDamageInflicted(Unit_Devastator) + getHouse()->getNumItemDamageInflicted(Unit_SonicTank) + getHouse()->getNumItemDamageInflicted(Unit_Deviator);
	int weightedNumLostSpecialUnits = (getHouse()->getNumLostItems(Unit_Devastator) * data[Unit_Devastator][houseID].price)
		+ (getHouse()->getNumLostItems(Unit_SonicTank) * data[Unit_SonicTank][houseID].price)
		+ (getHouse()->getNumLostItems(Unit_Deviator) * data[Unit_Deviator][houseID].price)
		+ 700; // middle ground 1 for special units
	FixPoint dlrSpecial = FixPoint(numSpecialUnitsDamageInflicted) / FixPoint(weightedNumLostSpecialUnits);
	FixPoint dlrLauncher = getHouse()->getNumItemDamageInflicted(Unit_Launcher) / FixPoint((1 + getHouse()->getNumLostItems(Unit_Launcher)) * data[Unit_Launcher][houseID].price);
	FixPoint dlrOrnithopter = getHouse()->getNumItemDamageInflicted(Unit_Ornithopter) / FixPoint((1 + getHouse()->getNumLostItems(Unit_Ornithopter)) * data[Unit_Ornithopter][houseID].price);

	Sint32 totalDamage = getHouse()->getNumItemDamageInflicted(Unit_Tank)
		+ getHouse()->getNumItemDamageInflicted(Unit_SiegeTank)
		+ getHouse()->getNumItemDamageInflicted(Unit_Devastator)
		+ getHouse()->getNumItemDamageInflicted(Unit_Launcher)
		+ getHouse()->getNumItemDamageInflicted(Unit_Ornithopter);

	// Harkonnen can't build ornithopers
	if (houseID == HOUSE_HARKONNEN) {
		dlrOrnithopter = 0;
	}

	// Ordos can't build Launchers
	if (houseID == HOUSE_ORDOS) {
		dlrLauncher = 0;
	}

	// Sonic tanks can get into negative damage territory
	if (dlrSpecial < 0) {
		dlrSpecial = 0;
	}

	FixPoint dlrTotal = dlrTank + dlrSiege + dlrSpecial + dlrLauncher + dlrOrnithopter;

	if (dlrTotal < 0) {
		dlrTotal = 0;
	}

	logDebug("Dmg: %d DLR: %f", totalDamage, dlrTotal.toFloat());

	/// Calculate ratios of launcher, special and light tanks. Remainder will be tank
	FixPoint launcherPercent = dlrLauncher / dlrTotal;
	FixPoint specialPercent = dlrSpecial / dlrTotal;
	FixPoint siegePercent = dlrSiege / dlrTotal;
	FixPoint ornithopterPercent = dlrOrnithopter / dlrTotal;
	FixPoint tankPercent = dlrTank / dlrTotal;



	// If we haven't done much damage just keep all ratios at optimised defaults
	// These ratios are based on end game stats over a number of AI test runs to see
	// Which units perform. By and large launchers and siege tanks have the best damage to loss ratio
	// commenting this out for now

	// Commenting ignoring the logic for now and going with gut feel
	if (totalDamage < 3000) {
		switch (houseID) {
		case HOUSE_HARKONNEN:
			launcherPercent = 0.50_fix;  // Reduced from 0.70 to cap at 50%
			specialPercent = 0.15_fix;   // Increased to compensate
			siegePercent = 0.15_fix;     // Increased to compensate
			tankPercent = 0.20_fix;      // Added tank allocation
			ornithopterPercent = 0.0_fix;
			break;

		case HOUSE_ORDOS:
			launcherPercent = 0.0_fix; // Don't have these
			specialPercent = 0.25_fix;
			siegePercent = 0.25_fix;
			tankPercent = 0.25_fix;
			ornithopterPercent = 0.25_fix;
			break;

		case HOUSE_ATREIDES:
			launcherPercent = 0.25_fix;  // Increased from 0.20
			specialPercent = 0.50_fix;   // Reduced from 0.65 to cap at 50%
			siegePercent = 0.10_fix;     // Increased from 0.00
			tankPercent = 0.10_fix;      // Increased from 0.00
			ornithopterPercent = 0.05_fix; // Reduced from 0.15
			break;
		
		case HOUSE_FREMEN:
			launcherPercent = 0.20_fix;
			specialPercent = 0.00_fix;
			siegePercent = 0.15_fix;     // Increased from 0.05
			tankPercent = 0.50_fix;      // Reduced from 0.65 to cap at 50%
			ornithopterPercent = 0.15_fix; // Increased from 0.10
			break;
		
		case HOUSE_SARDAUKAR:
			launcherPercent = 0.45_fix;
			specialPercent = 0.05_fix;   // Increased from 0.00
			siegePercent = 0.40_fix;
			tankPercent = 0.05_fix;
			ornithopterPercent = 0.05_fix; // Reduced from 0.10
		break;

		default:
			launcherPercent = 0.30_fix;
			specialPercent = 0.05_fix;
			siegePercent = 0.30_fix;
			tankPercent = 0.30_fix;
			ornithopterPercent = 0.05_fix; // Reduced from 0.10

			break;
		}
	}

	// Apply 60% cap to all unit type allocations (both default and DLR-based)
	const FixPoint maxAllocation = 0.60_fix;
	
	// Store original percentages for redistribution
	FixPoint originalTotal = launcherPercent + specialPercent + siegePercent + tankPercent + ornithopterPercent;
	
	// Cap each percentage at 60%
	bool anyWasCapped = false;
	if (launcherPercent > maxAllocation) {
		launcherPercent = maxAllocation;
		anyWasCapped = true;
	}
	if (specialPercent > maxAllocation) {
		specialPercent = maxAllocation;
		anyWasCapped = true;
	}
	if (siegePercent > maxAllocation) {
		siegePercent = maxAllocation;
		anyWasCapped = true;
	}
	if (tankPercent > maxAllocation) {
		tankPercent = maxAllocation;
		anyWasCapped = true;
	}
	if (ornithopterPercent > maxAllocation) {
		ornithopterPercent = maxAllocation;
		anyWasCapped = true;
	}
	
	// If any unit was capped, redistribute the remaining percentage
	if (anyWasCapped && originalTotal > 0) {
		FixPoint cappedTotal = launcherPercent + specialPercent + siegePercent + tankPercent + ornithopterPercent;
		FixPoint remainingPercent = 1.0_fix - cappedTotal;
		
		// Calculate total DLR of non-capped unit types for proportional distribution
		FixPoint uncappedDlrTotal = 0;
		if (launcherPercent < maxAllocation) uncappedDlrTotal += dlrLauncher;
		if (specialPercent < maxAllocation) uncappedDlrTotal += dlrSpecial;
		if (siegePercent < maxAllocation) uncappedDlrTotal += dlrSiege;
		if (tankPercent < maxAllocation) uncappedDlrTotal += dlrTank;
		if (ornithopterPercent < maxAllocation) uncappedDlrTotal += dlrOrnithopter;
		
		// Distribute remaining percentage proportionally based on damage performance
		if (uncappedDlrTotal > 0 && remainingPercent > 0) {
			if (launcherPercent < maxAllocation) {
				FixPoint proportionalShare = (dlrLauncher / uncappedDlrTotal) * remainingPercent;
				launcherPercent = std::min(maxAllocation, launcherPercent + proportionalShare);
			}
			if (specialPercent < maxAllocation) {
				FixPoint proportionalShare = (dlrSpecial / uncappedDlrTotal) * remainingPercent;
				specialPercent = std::min(maxAllocation, specialPercent + proportionalShare);
			}
			if (siegePercent < maxAllocation) {
				FixPoint proportionalShare = (dlrSiege / uncappedDlrTotal) * remainingPercent;
				siegePercent = std::min(maxAllocation, siegePercent + proportionalShare);
			}
			if (tankPercent < maxAllocation) {
				FixPoint proportionalShare = (dlrTank / uncappedDlrTotal) * remainingPercent;
				tankPercent = std::min(maxAllocation, tankPercent + proportionalShare);
			}
			if (ornithopterPercent < maxAllocation) {
				FixPoint proportionalShare = (dlrOrnithopter / uncappedDlrTotal) * remainingPercent;
				ornithopterPercent = std::min(maxAllocation, ornithopterPercent + proportionalShare);
			}
		} else if (remainingPercent > 0) {
			// Fallback: if no unit has done damage yet, distribute equally
			int availableTypes = 0;
			if (launcherPercent < maxAllocation) availableTypes++;
			if (specialPercent < maxAllocation) availableTypes++;
			if (siegePercent < maxAllocation) availableTypes++;
			if (tankPercent < maxAllocation) availableTypes++;
			if (ornithopterPercent < maxAllocation) availableTypes++;
			
			if (availableTypes > 0) {
				FixPoint bonusPerType = remainingPercent / FixPoint(availableTypes);
				if (launcherPercent < maxAllocation) {
					launcherPercent = std::min(maxAllocation, launcherPercent + bonusPerType);
				}
				if (specialPercent < maxAllocation) {
					specialPercent = std::min(maxAllocation, specialPercent + bonusPerType);
				}
				if (siegePercent < maxAllocation) {
					siegePercent = std::min(maxAllocation, siegePercent + bonusPerType);
				}
				if (tankPercent < maxAllocation) {
					tankPercent = std::min(maxAllocation, tankPercent + bonusPerType);
				}
				if (ornithopterPercent < maxAllocation) {
					ornithopterPercent = std::min(maxAllocation, ornithopterPercent + bonusPerType);
				}
			}
		}
	}
	
	// Final normalization only if total is significantly different from 1.0
	FixPoint finalTotal = launcherPercent + specialPercent + siegePercent + tankPercent + ornithopterPercent;
	if (finalTotal > 0 && (finalTotal < 0.95_fix || finalTotal > 1.05_fix)) {
		launcherPercent = launcherPercent / finalTotal;
		specialPercent = specialPercent / finalTotal;
		siegePercent = siegePercent / finalTotal;
		tankPercent = tankPercent / finalTotal;
		ornithopterPercent = ornithopterPercent / finalTotal;
	}
	
	// Ensure ornithopters always get at least 5% allocation (except for Harkonnen who can't build them)
	const FixPoint ornithopterMinimum = 0.05_fix;
	if (houseID != HOUSE_HARKONNEN && ornithopterPercent < ornithopterMinimum) {
		FixPoint deficit = ornithopterMinimum - ornithopterPercent;
		ornithopterPercent = ornithopterMinimum;
		
		// Reduce other unit types proportionally to compensate
		FixPoint otherTotal = launcherPercent + specialPercent + siegePercent + tankPercent;
		if (otherTotal > deficit) {
			FixPoint reductionRatio = (otherTotal - deficit) / otherTotal;
			launcherPercent *= reductionRatio;
			specialPercent *= reductionRatio;
			siegePercent *= reductionRatio;
			tankPercent *= reductionRatio;
		}
	}

	// lets analyse damage inflicted

	logDebug("  Tank: %d/%d %f Siege: %d/%d %f Special: %d/%d %f Launch: %d/%d %f Orni: %d/%d %f",
		getHouse()->getNumItemDamageInflicted(Unit_Tank), getHouse()->getNumLostItems(Unit_Tank) * 300, tankPercent.toDouble(),
		getHouse()->getNumItemDamageInflicted(Unit_SiegeTank), getHouse()->getNumLostItems(Unit_SiegeTank) * 600, siegePercent.toDouble(),
		getHouse()->getNumItemDamageInflicted(Unit_SonicTank) + getHouse()->getNumItemDamageInflicted(Unit_Devastator) + getHouse()->getNumItemDamageInflicted(Unit_Deviator),
		getHouse()->getNumLostItems(Unit_SonicTank) * 600 + getHouse()->getNumLostItems(Unit_Devastator) * 800 + getHouse()->getNumLostItems(Unit_Deviator) * 750,
		specialPercent.toDouble(),
		getHouse()->getNumItemDamageInflicted(Unit_Launcher), getHouse()->getNumLostItems(Unit_Launcher) * 450, launcherPercent.toDouble(),
		getHouse()->getNumItemDamageInflicted(Unit_Ornithopter), getHouse()->getNumLostItems(Unit_Ornithopter) * data[Unit_Ornithopter][houseID].price, ornithopterPercent.toDouble()
	);
	
	// End of adaptive unit prioritisation algorithm


	for (const StructureBase* pStructure : getStructureList()) {
		if (pStructure->getOwner() == getHouse()) {
			if ((pStructure->isRepairing() == false)
				&& (pStructure->getHealth() < pStructure->getMaxHealth())
				&& (!getGameInitSettings().getGameOptions().concreteRequired
					|| pStructure->getItemID() == Structure_Palace) // Palace repairs for free
				&& (pStructure->getItemID() != Structure_Refinery
					&& pStructure->getItemID() != Structure_Silo
					&& pStructure->getItemID() != Structure_Radar
					&& pStructure->getItemID() != Structure_WindTrap))
			{
				doRepair(pStructure);
			}
			else if ((pStructure->isRepairing() == false)
				&& (pStructure->getHealth() < pStructure->getMaxHealth() * 0.40_fix)
				&& money > 1000) {
				doRepair(pStructure);
			}
			else if ((pStructure->isRepairing() == false) && money > 5000) {
				// Repair if we are rich
				doRepair(pStructure);
			}
			else if (pStructure->getItemID() == Structure_RocketTurret) {
				if (!getGameInitSettings().getGameOptions().structuresDegradeOnConcrete || pStructure->hasATarget()) {
					doRepair(pStructure);
				}
			}

			// Special weapon launch logic
			if (pStructure->getItemID() == Structure_Palace) {

				const Palace* pPalace = static_cast<const Palace*>(pStructure);
				if (pPalace->isSpecialWeaponReady()) {

					if (houseID != HOUSE_HARKONNEN && houseID != HOUSE_SARDAUKAR) {
						doSpecialWeapon(pPalace);
					}
					else {
						int enemyHouseID = -1;
						int enemyHouseBuildingCount = 0;

						for (int i = 0; i < NUM_HOUSES; i++) {
							if (getHouse(i) != nullptr) {
								if (getHouse(i)->getTeamID() != getHouse()->getTeamID() && getHouse(i)->getNumStructures() > enemyHouseBuildingCount) {
									enemyHouseBuildingCount = getHouse(i)->getNumStructures();
									enemyHouseID = i;
								}
							}
						}

						if ((enemyHouseID != -1) && (houseID == HOUSE_HARKONNEN || houseID == HOUSE_SARDAUKAR)) {
							Coord target = findBaseCentre(enemyHouseID);
							doLaunchDeathhand(pPalace, target.x, target.y);
						}
					}
				}
			}

			if (pStructure->isABuilder()) {
				const BuilderBase* pBuilder = static_cast<const BuilderBase*>(pStructure);
				switch (pStructure->getItemID()) {

				case Structure_LightFactory: {
					if (!pBuilder->isUpgrading()
						&& gameMode == GameMode::Campaign
						&& money > 1000
						&& itemCount[Structure_HeavyFactory] == 0  // Only build light units if no Heavy Factory
						&& pBuilder->getProductionQueueSize() < 1
						&& pBuilder->getBuildListSize() > 0
						&& militaryValue < militaryValueLimit) {

						if (pBuilder->getCurrentUpgradeLevel() < pBuilder->getMaxUpgradeLevel() && getHouse()->getCredits() > 1500) {
							doUpgrade(pBuilder);
						}
						else if (!getHouse()->isGroundUnitLimitReached()) {
							Uint32 itemID = NONE_ID;

							if (pBuilder->isAvailableToBuild(Unit_RaiderTrike)) {
								itemID = Unit_RaiderTrike;
							}
							else if (pBuilder->isAvailableToBuild(Unit_Quad)) {
								itemID = Unit_Quad;
							}
							else if (pBuilder->isAvailableToBuild(Unit_Trike)) {
								itemID = Unit_Trike;
							}

							if (itemID != NONE_ID) {
								doProduceItem(pBuilder, itemID);
								itemCount[itemID]++;
							}
						}
					}
				} break;

				case Structure_WOR: {
					if (!pBuilder->isUpgrading()
						&& pBuilder->isAvailableToBuild(Unit_Trooper)
						&& gameMode == GameMode::Campaign
						&& money > 1000
						&& itemCount[Structure_HeavyFactory] == 0  // Only build troopers if no Heavy Factory
						&& pBuilder->getProductionQueueSize() < 1
						&& pBuilder->getBuildListSize() > 0
						&& !getHouse()->isInfantryUnitLimitReached()
						&& militaryValue < militaryValueLimit) {

						doProduceItem(pBuilder, Unit_Trooper);
						itemCount[Unit_Trooper]++;
					}
				} break;

				case Structure_Barracks: {
					if (!pBuilder->isUpgrading()
						&& pBuilder->isAvailableToBuild(Unit_Soldier)
						&& gameMode == GameMode::Campaign
						&& itemCount[Structure_HeavyFactory] == 0  // Only build soldiers if no Heavy Factory
						&& itemCount[Structure_WOR] == 0
						&& money > 1000
						&& pBuilder->getProductionQueueSize() < 1
						&& pBuilder->getBuildListSize() > 0
						&& !getHouse()->isInfantryUnitLimitReached()
						&& militaryValue < militaryValueLimit) {

						doProduceItem(pBuilder, Unit_Soldier);
						itemCount[Unit_Soldier]++;
					}
				} break;

				case Structure_HighTechFactory: {
					int ornithopterValue = data[Unit_Ornithopter][houseID].price * itemCount[Unit_Ornithopter];

					// Build carryalls using simple formula: (military value / 3000) + (harvesters / 3)
					int totalCarryalls = itemCount[Unit_Carryall];
					int neededCarryalls = std::max(1, (militaryValue / 3000) + (itemCount[Unit_Harvester] / 3));
					
					if (pBuilder->isAvailableToBuild(Unit_Carryall)
						&& totalCarryalls < neededCarryalls
						&& (pBuilder->getProductionQueueSize() < 1)
						&& money > 1000
						&& !getHouse()->isAirUnitLimitReached()) {
						doProduceItem(pBuilder, Unit_Carryall);
						itemCount[Unit_Carryall]++;
						logDebug("Build Carryall: have %d, need %d (military: %d, harvesters: %d)", totalCarryalls, neededCarryalls, militaryValue, itemCount[Unit_Harvester]);
					}
					else if ((money > 500) && (pBuilder->isUpgrading() == false) && (pBuilder->getCurrentUpgradeLevel() < pBuilder->getMaxUpgradeLevel())) {
						if (pBuilder->getHealth() >= pBuilder->getMaxHealth()) {
							doUpgrade(pBuilder);
						}
						else {
							doRepair(pBuilder);
						}
					}
					else if (pBuilder->isAvailableToBuild(Unit_Ornithopter)
						&& (militaryValue * ornithopterPercent > ornithopterValue)
						&& (pBuilder->getProductionQueueSize() < 1)
						&& !getHouse()->isAirUnitLimitReached()
						&& money > 1200) {
						// Current value and what percentage of military we want used to determine
						// whether to build an additional unit.
						doProduceItem(pBuilder, Unit_Ornithopter);
						itemCount[Unit_Ornithopter]++;
						money -= data[Unit_Ornithopter][houseID].price;
						militaryValue += data[Unit_Ornithopter][houseID].price;
					}
				} break;

				case Structure_HeavyFactory: {
					// only if the factory isn't busy
					if ((pBuilder->isUpgrading() == false) && (pBuilder->getProductionQueueSize() < 1) && (pBuilder->getBuildListSize() > 0)) {
						// we need a construction yard. Build an MCV if we don't have a starport
						if ((difficulty == Difficulty::Hard || difficulty == Difficulty::Brutal)
							&& itemCount[Unit_MCV] + itemCount[Structure_ConstructionYard] + itemCount[Structure_StarPort] < 1
							&& pBuilder->isAvailableToBuild(Unit_MCV)
							&& !getHouse()->isGroundUnitLimitReached()) {
							doProduceItem(pBuilder, Unit_MCV);
							itemCount[Unit_MCV]++;
						}
						else if ((money > 10000) && (pBuilder->isUpgrading() == false) && (pBuilder->getCurrentUpgradeLevel() < pBuilder->getMaxUpgradeLevel())) {
							if (pBuilder->getHealth() >= pBuilder->getMaxHealth()) {
								doUpgrade(pBuilder);
							}
							else {
								doRepair(pBuilder);
							}
						}
						else if (gameMode == GameMode::Custom && (itemCount[Structure_ConstructionYard] + itemCount[Unit_MCV]) * 10000 < money
							&& pBuilder->isAvailableToBuild(Unit_MCV)
							&& itemCount[Structure_ConstructionYard] + itemCount[Unit_MCV] < 4
							&& !getHouse()->isGroundUnitLimitReached()) {
							// If we are really rich, like in all against Atriedes
							doProduceItem(pBuilder, Unit_MCV);
							itemCount[Unit_MCV]++;
						}
						else if (gameMode == GameMode::Custom
							&& pBuilder->isAvailableToBuild(Unit_Harvester)
							&& !getHouse()->isGroundUnitLimitReached()
							&& itemCount[Unit_Harvester] < militaryValue / 1000
							&& itemCount[Unit_Harvester] < harvesterLimit) {
							// In case we get given lots of money, it will eventually run out so we need to be prepared
							doProduceItem(pBuilder, Unit_Harvester);
							itemCount[Unit_Harvester]++;
						}
						else if (itemCount[Unit_Harvester] < harvesterLimit
							&& pBuilder->isAvailableToBuild(Unit_Harvester)
							&& !getHouse()->isGroundUnitLimitReached()
							&& (money < 2000 || gameMode == GameMode::Campaign)) {
							//logDebug("*Building a Harvester.",
							//itemCount[Unit_Harvester], harvesterLimit, money);
							doProduceItem(pBuilder, Unit_Harvester);
							itemCount[Unit_Harvester]++;
						}
						else if ((money > 500) && (pBuilder->isUpgrading() == false) && (pBuilder->getCurrentUpgradeLevel() < pBuilder->getMaxUpgradeLevel())) {
							if (pBuilder->getHealth() >= pBuilder->getMaxHealth()) {
								doUpgrade(pBuilder);
							}
							else {
								doRepair(pBuilder);
							}
						}
						else if (money > 1200 && militaryValue < militaryValueLimit && !getHouse()->isGroundUnitLimitReached()) {
							// TODO: This entire section needs to be refactored to make it more generic
							// Limit enemy military units based on difficulty

							// Calculate current value of units
							int launcherValue = data[Unit_Launcher][houseID].price * itemCount[Unit_Launcher];
							int specialValue = data[Unit_Devastator][houseID].price * itemCount[Unit_Devastator]
								+ data[Unit_Deviator][houseID].price * itemCount[Unit_Deviator]
								+ data[Unit_SonicTank][houseID].price * itemCount[Unit_SonicTank];
							int siegeValue = data[Unit_SiegeTank][houseID].price * itemCount[Unit_SiegeTank];


							/// Use current value and what percentage of military we want to determine
							/// whether to build an additional unit.
							if (pBuilder->isAvailableToBuild(Unit_Launcher) && (militaryValue * launcherPercent > launcherValue)) {
								doProduceItem(pBuilder, Unit_Launcher);
								itemCount[Unit_Launcher]++;
								money -= data[Unit_Launcher][houseID].price;
								militaryValue += data[Unit_Launcher][houseID].price;
							}
							else if (pBuilder->isAvailableToBuild(Unit_Devastator) && (militaryValue * specialPercent > specialValue)) {
								doProduceItem(pBuilder, Unit_Devastator);
								itemCount[Unit_Devastator]++;
								money -= data[Unit_Devastator][houseID].price;
								militaryValue += data[Unit_Devastator][houseID].price;
							}
							else if (pBuilder->isAvailableToBuild(Unit_SonicTank) && (militaryValue * specialPercent > specialValue)) {
								doProduceItem(pBuilder, Unit_SonicTank);
								itemCount[Unit_SonicTank]++;
								money -= data[Unit_SonicTank][houseID].price;
								militaryValue += data[Unit_SonicTank][houseID].price;
							}
							else if (pBuilder->isAvailableToBuild(Unit_Deviator) && (militaryValue * specialPercent > specialValue)) {
								doProduceItem(pBuilder, Unit_Deviator);
								itemCount[Unit_Deviator]++;
								money -= data[Unit_Deviator][houseID].price;
								militaryValue += data[Unit_Deviator][houseID].price;
							}
							else if (pBuilder->isAvailableToBuild(Unit_SiegeTank) && (militaryValue * siegePercent > siegeValue)) {
								doProduceItem(pBuilder, Unit_SiegeTank);
								itemCount[Unit_SiegeTank]++;
								money -= data[Unit_Tank][houseID].price;
								militaryValue += data[Unit_SiegeTank][houseID].price;
							}
							else if (pBuilder->isAvailableToBuild(Unit_Tank)) {
								// Tanks for all else
								doProduceItem(pBuilder, Unit_Tank);
								itemCount[Unit_Tank]++;
								money -= data[Unit_Tank][houseID].price;
								militaryValue += data[Unit_Tank][houseID].price;
							}
						}
					}

				} break;

				case Structure_StarPort: {
					const StarPort* pStarPort = static_cast<const StarPort*>(pBuilder);
					if (pStarPort->okToOrder()) {
						const Choam& choam = getHouse()->getChoam();

						// We need a construction yard!!
						if ((difficulty == Difficulty::Hard || difficulty == Difficulty::Brutal)
							&& pStarPort->isAvailableToBuild(Unit_MCV)
							&& choam.getNumAvailable(Unit_MCV) > 0
							&& itemCount[Structure_ConstructionYard] + itemCount[Unit_MCV] < 1) {
							doProduceItem(pBuilder, Unit_MCV);
							itemCount[Unit_MCV]++;
							money = money - choam.getPrice(Unit_MCV);
						}

						if (money > choam.getPrice(Unit_Carryall) && choam.getNumAvailable(Unit_Carryall) > 0 && itemCount[Unit_Carryall] == 0) {
							// Get at least one Carryall
							doProduceItem(pBuilder, Unit_Carryall);
							itemCount[Unit_Carryall]++;
							money = money - choam.getPrice(Unit_Carryall);
						}

						while (money > choam.getPrice(Unit_Harvester) && choam.getNumAvailable(Unit_Harvester) > 0 && itemCount[Unit_Harvester] < harvesterLimit) {
							doProduceItem(pBuilder, Unit_Harvester);
							itemCount[Unit_Harvester]++;
							money = money - choam.getPrice(Unit_Harvester);
						}

						int itemCountUnits = itemCount[Unit_Tank] + itemCount[Unit_SiegeTank] + itemCount[Unit_Launcher] + itemCount[Unit_Harvester];

						// Only buy carryalls from StarPort if we have very few (let High-Tech Factory handle main production)
						while (money > choam.getPrice(Unit_Carryall) && choam.getNumAvailable(Unit_Carryall) > 0 && itemCount[Unit_Carryall] < 2) {
							doProduceItem(pBuilder, Unit_Carryall);
							itemCount[Unit_Carryall]++;
							money = money - choam.getPrice(Unit_Carryall);
						}

						while (militaryValue < militaryValueLimit && money > choam.getPrice(Unit_SiegeTank) && choam.getNumAvailable(Unit_SiegeTank) > 0
							&& choam.isCheap(Unit_SiegeTank) && militaryValue < militaryValueLimit && money > 2000) {
							doProduceItem(pBuilder, Unit_SiegeTank);
							itemCount[Unit_SiegeTank]++;
							money = money - choam.getPrice(Unit_SiegeTank);
							militaryValue += data[Unit_SiegeTank][houseID].price;
						}

						while (militaryValue < militaryValueLimit && money > choam.getPrice(Unit_Launcher) && choam.getNumAvailable(Unit_Launcher) > 0
							&& choam.isCheap(Unit_Launcher) && militaryValue < militaryValueLimit && money > 2000) {
							doProduceItem(pBuilder, Unit_Launcher);
							itemCount[Unit_Launcher]++;
							money = money - choam.getPrice(Unit_Launcher);
							militaryValue += data[Unit_Launcher][houseID].price;
						}

						while (militaryValue < militaryValueLimit && money > choam.getPrice(Unit_Tank) && choam.getNumAvailable(Unit_Tank) > 0
							&& choam.isCheap(Unit_Tank) && militaryValue < militaryValueLimit && money > 2000) {
							doProduceItem(pBuilder, Unit_Tank);
							itemCount[Unit_Tank]++;
							money = money - choam.getPrice(Unit_Tank);
							militaryValue += data[Unit_Tank][houseID].price;
						}



						while (militaryValue < militaryValueLimit && money > choam.getPrice(Unit_Ornithopter) && choam.getNumAvailable(Unit_Ornithopter) > 0
							&& choam.isCheap(Unit_Ornithopter) && militaryValue < militaryValueLimit && money > 2000) {
							doProduceItem(pBuilder, Unit_Ornithopter);
							itemCount[Unit_Ornithopter]++;
							money = money - choam.getPrice(Unit_Ornithopter);
							militaryValue += data[Unit_Ornithopter][houseID].price;
						}



						doPlaceOrder(pStarPort);
					}

				} break;

				case Structure_ConstructionYard: {

					// If rocket turrets don't need power then let's build some for defense
					int rocketTurretValue = itemCount[Structure_RocketTurret] * 250;

					// Only build rocket turrets if they don't need power (or we have power)
					if (getGameInitSettings().getGameOptions().rocketTurretsNeedPower && !getHouse()->hasPower()) {
						rocketTurretValue = 1000000; // If rocket turrets need power and we don't have it, don't build them
					}

					const ConstructionYard* pConstYard = static_cast<const ConstructionYard*>(pBuilder);

					if (!pBuilder->isUpgrading() && getHouse()->getCredits() > 100 && (pBuilder->getProductionQueueSize() < 1) && pBuilder->getBuildListSize()) {

						// Campaign Build order, iterate through the buildings, if the number that exist
						// is less than the number that should exist, then build the one that is missing

						if (gameMode == GameMode::Campaign && difficulty != Difficulty::Brutal) {
							//logDebug("GameMode Campaign.. ");

							for (int i = Structure_FirstID; i <= Structure_LastID; i++) {
								if (itemCount[i] < initialItemCount[i]
									&& pBuilder->isAvailableToBuild(i)
									&& findPlaceLocation(i).isValid()
									&& !pBuilder->isUpgrading()
									&& pBuilder->getProductionQueueSize() < 1) {

									logDebug("***CampAI Build itemID: %o structure count: %o, initial count: %o", i, itemCount[i], initialItemCount[i]);
									doProduceItem(pBuilder, i);
									itemCount[i]++;
								}
							}

							// If Campaign AI can't build military, let it build up its cash reserves and defenses

							if (pStructure->getHealth() < pStructure->getMaxHealth()) {
								doRepair(pBuilder);
							}
							else if (pBuilder->getCurrentUpgradeLevel() < pBuilder->getMaxUpgradeLevel()
								&& !pBuilder->isUpgrading()
								&& itemCount[Unit_Harvester] >= harvesterLimit) {

								doUpgrade(pBuilder);
								logDebug("***CampAI Upgrade builder");
							}
							else if ((getHouse()->getProducedPower() < getHouse()->getPowerRequirement())
								&& pBuilder->isAvailableToBuild(Structure_WindTrap)
								&& itemCount[Structure_WindTrap] < initialItemCount[Structure_WindTrap]  // Only build up to initial count
								&& findPlaceLocation(Structure_WindTrap).isValid()
								&& pBuilder->getProductionQueueSize() == 0) {

								doProduceItem(pBuilder, Structure_WindTrap);

								logDebug("***CampAI Build A new Windtrap increasing count to: %d (max: %d)", itemCount[Structure_WindTrap], initialItemCount[Structure_WindTrap]);
							}
							else if ((getHouse()->getStoredCredits() > getHouse()->getCapacity() * 0.90_fix)  // Only build when 90% full
								&& pBuilder->isAvailableToBuild(Structure_Silo)
								&& findPlaceLocation(Structure_Silo).isValid()
								&& pBuilder->getProductionQueueSize() == 0) {

								doProduceItem(pBuilder, Structure_Silo);
								itemCount[Structure_Silo]++;

								logDebug("***CampAI Build A new Silo increasing count to: %d (credits: %d/%d)", itemCount[Structure_Silo], getHouse()->getStoredCredits().lround(), getHouse()->getCapacity());
							}
							else if (money > 3000
								&& pBuilder->isAvailableToBuild(Structure_RocketTurret)
								&& findPlaceLocation(Structure_RocketTurret).isValid()
								&& pBuilder->getProductionQueueSize() == 0
								&& (itemCount[Structure_RocketTurret] <
									(itemCount[Structure_Silo] + itemCount[Structure_Refinery]) * 2)) {

								doProduceItem(pBuilder, Structure_RocketTurret);

								logDebug("***CampAI Build A new Rocket turret increasing count to: %d", itemCount[Structure_RocketTurret]);
							}

							buildTimer = getRandomGen().rand(0, 3) * 5;
						}
						else {
							// custom AI starts here:

							Uint32 itemID = NONE_ID;

							// Count enemy ornithopters at the start
							int enemyOrnithopterCount = 0;
							for (int i = 0; i < NUM_HOUSES; i++) {
								const House* pHouse = currentGame->getHouse(i);
								if (pHouse && pHouse->getTeamID() != getHouse()->getTeamID()) {
									enemyOrnithopterCount += pHouse->getNumItems(Unit_Ornithopter);
								}
							}

							if (itemCount[Structure_WindTrap] == 0 && pBuilder->isAvailableToBuild(Structure_WindTrap)) {
								itemID = Structure_WindTrap;
							}
							else if ((itemCount[Structure_Refinery] == 0 || itemCount[Structure_Refinery] < itemCount[Unit_Harvester] / 3) && pBuilder->isAvailableToBuild(Structure_Refinery)) {
								itemID = Structure_Refinery;
								itemCount[Unit_Harvester]++;
							}
							else if (itemCount[Structure_Refinery] < 4 && pBuilder->isAvailableToBuild(Structure_Refinery) && money < 4000) {
								itemID = Structure_Refinery;
								itemCount[Unit_Harvester]++;
							}
							else if (itemCount[Structure_StarPort] == 0 && pBuilder->isAvailableToBuild(Structure_StarPort) && findPlaceLocation(Structure_StarPort).isValid()) {
								itemID = Structure_StarPort;
							}
							// Build essential infrastructure with reasonable credit requirements
							else if (itemCount[Structure_Radar] == 0 && pBuilder->isAvailableToBuild(Structure_Radar) && money > 500) {
								itemID = Structure_Radar; // Outpost for unit coordination
							}
							// Counter enemy ornithopters with rocket turrets or upgrades
							else if (enemyOrnithopterCount > itemCount[Structure_RocketTurret]) {
								if (pBuilder->getCurrentUpgradeLevel() < 2) {
									if (pBuilder->getHealth() < pBuilder->getMaxHealth() && !pBuilder->isRepairing()) {
										// Repair construction yard first if damaged
										doRepair(pBuilder);
										logDebug("COUNTER-ORNITHOPTER: Repairing construction yard - health low");
									}
									else if (pBuilder->getHealth() >= pBuilder->getMaxHealth() && !pBuilder->isUpgrading()) {
										// Upgrade construction yard to unlock rocket turrets
										doUpgrade(pBuilder);
										logDebug("COUNTER-ORNITHOPTER: Upgrading construction yard to level 2");
									}
								}
								else if (pBuilder->isAvailableToBuild(Structure_RocketTurret) 
									&& findPlaceLocation(Structure_RocketTurret).isValid()
									&& (!getGameInitSettings().getGameOptions().rocketTurretsNeedPower || getHouse()->hasPower())) {
									// Build rocket turret to counter ornithopters
									itemID = Structure_RocketTurret;
									logDebug("COUNTER-ORNITHOPTER: Building rocket turret - enemy ornis: %d, our turrets: %d", enemyOrnithopterCount, itemCount[Structure_RocketTurret]);
								}
							}
							else if (pBuilder->isAvailableToBuild(Structure_LightFactory)
								&& itemCount[Structure_LightFactory] == 0 && money > 500) {
								itemID = Structure_LightFactory; // Essential for basic units
							}
							else if (pBuilder->isAvailableToBuild(Structure_HeavyFactory)
								&& itemCount[Structure_HeavyFactory] == 0 && money > 1000) {
								itemID = Structure_HeavyFactory; // First heavy factory
								logDebug("Build first Heavy Factory... money: %d", money);
							}							
							else if (itemCount[Structure_RepairYard] == 0 && pBuilder->isAvailableToBuild(Structure_RepairYard) && money > 1000) {
								itemID = Structure_RepairYard; // Essential for unit maintenance
							}
							else if (pBuilder->isAvailableToBuild(Structure_Refinery)
								&& money < 4000
								&& itemCount[Unit_Harvester] < harvesterLimit) {
								itemID = Structure_Refinery;
								itemCount[Unit_Harvester]++;
														}
							// Upgrade Construction Yard to unlock rocket turrets and advanced units
							else if (pBuilder->getCurrentUpgradeLevel() < 2 
								&& !pBuilder->isUpgrading() 
								&& pBuilder->getHealth() >= pBuilder->getMaxHealth()
								&& money > 1000) {
									if (pBuilder->getHealth() < pBuilder->getMaxHealth() && !pBuilder->isRepairing()) {
										// Repair construction yard first if damaged
										doRepair(pBuilder);
										logDebug("COUNTER-ORNITHOPTER: Repairing construction yard - health low");
									}
									else if (pBuilder->getHealth() >= pBuilder->getMaxHealth() && !pBuilder->isUpgrading()) {
										// Upgrade construction yard to unlock rocket turrets
										doUpgrade(pBuilder);
										logDebug("COUNTER-ORNITHOPTER: Upgrading construction yard to level 2");
									}
							}
							// Progressive rocket turret building after starport (if they don't need power or we have power)
							else if (itemCount[Structure_RocketTurret] < 2
										&& pBuilder->isAvailableToBuild(Structure_RocketTurret)
										&& findPlaceLocation(Structure_RocketTurret).isValid()
										&& (!getGameInitSettings().getGameOptions().rocketTurretsNeedPower || getHouse()->hasPower())
										&& money > 1000) {
								itemID = Structure_RocketTurret;
							}
							else if (itemCount[Structure_HighTechFactory] == 0 && money > 1000) {
								if (pBuilder->isAvailableToBuild(Structure_HighTechFactory)) {
									itemID = Structure_HighTechFactory;
								}
							}
							// If we need more refinerys for our harvesters or we don't have a heavy factory
							else if (((itemCount[Structure_Refinery] * 3 < itemCount[Unit_Harvester])
								|| (currentGame->techLevel < 4 && itemCount[Unit_Harvester] < harvesterLimit))
									&& pBuilder->isAvailableToBuild(Structure_Refinery)) {
								itemID = Structure_Refinery;
								itemCount[Unit_Harvester]++;
					
							}
							else if (itemCount[Structure_IX] == 0 && pBuilder->isAvailableToBuild(Structure_IX) && money > 1000) {
								itemID = Structure_IX; // House of IX for special units (after essential production buildings)
							}
							// HIGH PRIORITY: Heavy factories when we have good economy and infrastructure
							else if (money > 3000 && pBuilder->isAvailableToBuild(Structure_HeavyFactory)
								&& itemCount[Structure_IX] >= 1 && itemCount[Structure_RepairYard] >= 1
								&& (activeHeavyFactoryCount >= itemCount[Structure_HeavyFactory] || itemCount[Structure_HeavyFactory] < money / 4000)) {
								// Build heavy factories when we have good credits, infrastructure, and need production capacity
								itemID = Structure_HeavyFactory;
								logDebug("PRIORITY Heavy Factory - active: %d  total: %d  money: %d  capacity_limit: %d", activeHeavyFactoryCount, getHouse()->getNumItems(Structure_HeavyFactory), money, money / 4000);
							}
							// If we need more refinerys for our harvesters or we don't have a heavy factory
							else if (((itemCount[Structure_Refinery] * 3.5_fix < harvesterLimit)
							|| (currentGame->techLevel < 4 && itemCount[Unit_Harvester] < harvesterLimit))
								&& pBuilder->isAvailableToBuild(Structure_Refinery)) {
								itemID = Structure_Refinery;
								itemCount[Unit_Harvester]++;
				
							}
							else if (pBuilder->isAvailableToBuild(Structure_RepairYard) && money > 2000
								&& itemCount[Structure_RepairYard] * 6000 < militaryValue) {
								// If we have a lot of troops get some repair facilities (1 per 6000 military value)
								itemID = Structure_RepairYard;
								logDebug("Build Repair Yard: have %d, need %d (military: %d)", itemCount[Structure_RepairYard], (militaryValue / 6000) + 1, militaryValue);

							}
							else if (money > 3000 && pBuilder->isAvailableToBuild(Structure_HighTechFactory)
								&& itemCount[Structure_HighTechFactory] > 0 && activeHighTechFactoryCount >= itemCount[Structure_HighTechFactory]) {
								// Build additional high tech factory if all existing ones are busy
								itemID = Structure_HighTechFactory;
							}
							else if (getHouse()->getStoredCredits() + 1000 > (itemCount[Structure_Refinery] + itemCount[Structure_Silo]) * 1000 && pBuilder->isAvailableToBuild(Structure_Silo)) {
								// We are running out of spice storage capacity
								itemID = Structure_Silo;
							}
							else if (money > 5000
								&& pBuilder->isAvailableToBuild(Structure_Palace)
								&& (itemCount[Structure_Palace] == 0 || !getGameInitSettings().getGameOptions().onlyOnePalace)
								&& itemCount[Structure_HeavyFactory] > 0
								&& itemCount[Structure_LightFactory] > 0) {
								// Build palace after having basic military infrastructure
								// Allow multiple palaces if game mode permits
								itemID = Structure_Palace;
							}

							if (pBuilder->isAvailableToBuild(itemID) && findPlaceLocation(itemID).isValid() && itemID != NONE_ID) {
								doProduceItem(pBuilder, itemID);
								itemCount[itemID]++;
							} else {
								// If we can't build the desired structure, try to expand buildable area with concrete slabs
								// This helps create more valid placement locations for future buildings
								if (pBuilder->isAvailableToBuild(Structure_Slab1)) {
									Coord slabLocation = findPlaceLocationSimple(Structure_Slab1);
									if (slabLocation.isValid()) {
										doProduceItem(pBuilder, Structure_Slab1);
										logDebug("Building concrete slab to expand buildable area at (%d,%d)", slabLocation.x, slabLocation.y);
									}
								}
							}

						}
					}

					if (pBuilder->isWaitingToPlace()) {
						Uint32 itemToBePlaced = pBuilder->getCurrentProducedItem();
						Coord location;
						
						// Use appropriate placement method based on item type
						if (itemToBePlaced == Structure_Slab1) {
							// For concrete slabs, use simple method to find any valid location
							location = findPlaceLocationSimple(itemToBePlaced);
						} else {
							// For other structures, use normal method that favors adjacency
							location = findPlaceLocation(itemToBePlaced);
						}

						if (location.isValid()) {
							doPlaceStructure(pConstYard, location.x, location.y);
						}
						else {
							logDebug("Failed to find placement location for item %d, cancelling", itemToBePlaced);
							doCancelItem(pConstYard, itemToBePlaced);
						}
					}
				} break;
				}
			}
		}
	}



	buildTimer = getRandomGen().rand(0, 3) * 5;
}


void QuantBot::scrambleUnitsAndDefend(const ObjectBase* pIntruder, int numUnits) {
	for (const UnitBase* pUnit : getUnitList()) {
		if (pUnit && pUnit->isRespondable() && (pUnit->getOwner() == getHouse())) {
			if (!pUnit->hasATarget() && !pUnit->wasForced()) {
				Uint32 itemID = pUnit->getItemID();
				if ((itemID != Unit_Harvester) && (pUnit->getItemID() != Unit_MCV) && (pUnit->getItemID() != Unit_Carryall)
					&& (pUnit->getItemID() != Unit_Frigate) && (pUnit->getItemID() != Unit_Saboteur) && (pUnit->getItemID() != Unit_Sandworm)) {

					doSetAttackMode(pUnit, AREAGUARD);

					if (pUnit->getItemID() == Unit_Launcher || pUnit->getItemID() == Unit_Deviator) {
						doAttackObject(pUnit, pIntruder, false);
					}
					else {
						doAttackObject(pUnit, pIntruder, true);
					}

					if (getGameInitSettings().getGameOptions().manualCarryallDrops
						&& pUnit->isVisible()
						&& pUnit->isAGroundUnit()
						&& (pUnit->getItemID() != Unit_Deviator)
						&& (pUnit->getItemID() != Unit_Launcher)
						&& (blockDistance(pUnit->getLocation(), pUnit->getDestination()) >= 10)
						&& (pUnit->getHealth() / pUnit->getMaxHealth() > BADLYDAMAGEDRATIO)) {

						doRequestCarryallDrop(static_cast<const GroundUnit*>(pUnit)); //do request carryall to defend unit
					}

					if (--numUnits == 0) {
						break;
					}
				}
			}
		}
	}
}


void QuantBot::attack(int militaryValue) {
	// Safety check: if our house is null (e.g., during game cleanup), don't attack
	if (getHouse() == nullptr) {
		return;
	}

	// reset attack timer for every 15s
	attackTimer = MILLI2CYCLES(15000);

	// if AI is set to only defend then don't attack
	if (difficulty == Difficulty::Defend) {
		logDebug("Don't attack. Defend only AI");
		return;
		// Set max attack squad sizes for campaigns, AI will only attack with this number of units
		// should move and refactor this to run once at start. Also should rework military value
	}

	// Campaign AI can attack if:
	// 1. It has been attacked first (campaignAIAttackFlag = true), OR
	// 2. The initial attack timer has expired (this method was called)
	// Since we're in attack(), the timer has already expired, so Campaign AI can proceed
	if (gameMode == GameMode::Campaign) {
		logDebug("Campaign AI attack check: campaignAIAttackFlag=%s, timer expired=true", 
			campaignAIAttackFlag ? "true" : "false");
	}

	// Ornithopter attacks are now handled individually in checkAllUnits() for better tactical flexibility

	// Main attack loop
	if (militaryValue < militaryValueLimit * 0.35_fix) {
		logDebug("Don't attack. Not enough troops: house: %d  dif: %d  mStr: %d  mLim: %d",
			getHouse()->getHouseID(), static_cast<Uint8>(difficulty), militaryValue, militaryValueLimit, attackTimer);
		return;
	}

	// Don't attack if you don't yet have a repair yard, if you are at least tech 5
	if (getHouse()->getNumItems(Structure_RepairYard) == 0 && currentGame && currentGame->techLevel > 4) {
		logDebug("Don't attack. Wait until you have a repair yard.");
		return;
	}

	int attackSquadSize = 0; // how many units AI will send in attack squad
	int maxAttackSquadSize = 70; // max units that AI can send
	
	// First count existing hunting units
	for (const UnitBase* pUnit : getUnitList()) {
		if (pUnit && pUnit->isRespondable()
			&& (pUnit->getOwner() == getHouse())
			&& pUnit->isActive()
			&& !pUnit->isBadlyDamaged()
			&& pUnit->getAttackMode() == HUNT
			&& pUnit->getItemID() != Unit_Harvester
			&& pUnit->getItemID() != Unit_MCV
			&& pUnit->getItemID() != Unit_Carryall
			&& pUnit->getItemID() != Unit_Ornithopter
			&& pUnit->getItemID() != Unit_Sandworm) {
			attackSquadSize++;
		}
	}

	logDebug("Attack: house: %d  dif: %d  mStr: %d  mLim: %d  attackTimer: %d",
		getHouse()->getHouseID(), static_cast<Uint8>(difficulty), militaryValue, militaryValueLimit, attackTimer);

	Coord squadCenterLocation = findSquadCenter(getHouse()->getHouseID());

	for (const UnitBase* pUnit : getUnitList()) {
		if (pUnit && pUnit->isRespondable()
			&& (pUnit->getOwner() == getHouse())
			&& pUnit->isActive()
			&& !pUnit->isBadlyDamaged()
			&& !pUnit->wasForced()
			&& pUnit->getAttackMode() != RETREAT
			&& pUnit->getAttackMode() != HUNT  // Don't add units that are already hunting
			&& pUnit->getItemID() != Unit_Harvester
			&& pUnit->getItemID() != Unit_MCV
			&& pUnit->getItemID() != Unit_Carryall
			&& pUnit->getItemID() != Unit_Ornithopter
			&& pUnit->getItemID() != Unit_Sandworm)

		{	
			if (attackSquadSize >= maxAttackSquadSize) {
				logDebug("Attacking with %d units", attackSquadSize);
				return; // return if we have reached the squad size for the map
			}
			else {
				doSetAttackMode(pUnit, HUNT);
				attackSquadSize++;
			}
		}
	}
	logDebug("Attacking with %d units", attackSquadSize);

}


Coord QuantBot::findSquadRallyLocation() {
	int buildingCount = 0;
	int totalX = 0;
	int totalY = 0;

	// Find our base center
	for (const StructureBase* pCurrentStructure : getStructureList()) {
		if (pCurrentStructure->getOwner()->getHouseID() == getHouse()->getHouseID()) {
			buildingCount++;
			totalX += pCurrentStructure->getX();
			totalY += pCurrentStructure->getY();
		}
	}

	if (buildingCount == 0) {
		return Coord::Invalid();
	}

	Coord baseCentreLocation(totalX / buildingCount, totalY / buildingCount);

	// Find nearest sand tile from base center
	Coord nearestSandLocation = Coord::Invalid();
	FixPoint closestDistance = FixPt_MAX;

	// Safety check: ensure currentGameMap is not null
	if (currentGameMap == nullptr) {
		return baseCentreLocation; // Fallback to base center if map is not available
	}

	for (int y = 0; y < currentGameMap->getSizeY(); y++) {
		for (int x = 0; x < currentGameMap->getSizeX(); x++) {
			if (!currentGameMap->getTile(x, y)->isRock() && !currentGameMap->getTile(x, y)->hasSpice()) {
				// This is a sand tile without spice
				Coord sandCoord(x, y);
				FixPoint distance = blockDistance(baseCentreLocation, sandCoord);
				
				if (distance < closestDistance) {
					closestDistance = distance;
					nearestSandLocation = sandCoord;
				}
			}
		}
	}

	//logDebug("Squad rally location: %d, %d", nearestSandLocation.x , nearestSandLocation.y );

	return nearestSandLocation;
}

Coord QuantBot::findSquadRetreatLocation() {
	Coord newSquadRetreatLocation = Coord::Invalid();

	FixPoint closestDistance = FixPt_MAX;
	for (const StructureBase* pStructure : getStructureList()) {
		// if it is our building, check to see if it is closer to the squad rally point then we are
		if (pStructure->getOwner()->getHouseID() == getHouse()->getHouseID()) {
			Coord closestStructurePoint = pStructure->getClosestPoint(squadRallyLocation);
			FixPoint structureDistance = blockDistance(squadRallyLocation, closestStructurePoint);

			if (structureDistance < closestDistance) {
				closestDistance = structureDistance;
				newSquadRetreatLocation = closestStructurePoint;
			}
		}
	}

	return newSquadRetreatLocation;
}

Coord QuantBot::findTacticalRetreatLocation(const UnitBase* pUnit, const ObjectBase* pTarget) {
	if (!pUnit || !pTarget || !currentGame || !currentGameMap) {
		return Coord::Invalid();
	}
	
	// Get unit weapon range - add null safety checks
	if (!pUnit->getOwner()) {
		return Coord::Invalid();
	}
	int weaponRange = currentGame->objectData.data[pUnit->getItemID()][pUnit->getOwner()->getHouseID()].weaponrange;
	if (weaponRange <= 0) {
		return Coord::Invalid(); // Unit has no ranged weapon
	}
	
	Coord unitPos = pUnit->getLocation();
	Coord targetPos = pTarget->getLocation();
	
	// Calculate direction vector (from target to unit - opposite direction for retreat)
	int dirX = unitPos.x - targetPos.x;
	int dirY = unitPos.y - targetPos.y;
	
	// Check if units are on same or adjacent tiles
	if (dirX == 0 && dirY == 0) {
		// Units are on same tile, retreat to squad center
		return squadRallyLocation;
	}
	
	// Normalize direction (simple approach - use dominant direction)
	FixPoint normalizedX = 0;
	FixPoint normalizedY = 0;
	
	if (abs(dirX) > abs(dirY)) {
		// X direction dominates
		normalizedX = (dirX > 0) ? 1.0_fix : -1.0_fix;
		normalizedY = (dirY != 0) ? ((dirY > 0) ? 0.5_fix : -0.5_fix) : 0;
	} else {
		// Y direction dominates
		normalizedY = (dirY > 0) ? 1.0_fix : -1.0_fix;
		normalizedX = (dirX != 0) ? ((dirX > 0) ? 0.5_fix : -0.5_fix) : 0;
	}
	
	// Calculate optimal retreat distance (stay within weapon range but at safe distance)
	int safeDistance = weaponRange - 1; // 1 tile buffer for safety
	if (safeDistance < 3) safeDistance = 3; // Minimum safe distance
	
	// Calculate ideal retreat position
	FixPoint optimalDistance = FixPoint(safeDistance);
	FixPoint retreatX = FixPoint(targetPos.x) + (normalizedX * optimalDistance);
	FixPoint retreatY = FixPoint(targetPos.y) + (normalizedY * optimalDistance);
	
	Coord idealRetreatPos(retreatX.lround(), retreatY.lround());
	
	// Phase 1: Try to find safe position within weapon range
	const int maxSearchRadius = 8;
	for (int searchRadius = 0; searchRadius <= maxSearchRadius; searchRadius++) {
		for (int dx = -searchRadius; dx <= searchRadius; dx++) {
			for (int dy = -searchRadius; dy <= searchRadius; dy++) {
				// Only check tiles on the perimeter of current search radius
				if (searchRadius > 0 && abs(dx) != searchRadius && abs(dy) != searchRadius) {
					continue;
				}
				
				Coord candidatePos(idealRetreatPos.x + dx, idealRetreatPos.y + dy);
				
				// Check if tile exists and is passable
				if (!getMap().tileExists(candidatePos.x, candidatePos.y)) {
					continue;
				}
				
				if (!getMap().okayToPlaceStructure(candidatePos.x, candidatePos.y, 1, 1, false, nullptr)) {
					continue; // Tile not passable or occupied
				}
				
				// Calculate actual distance to target
				FixPoint actualDistance = blockDistance(candidatePos, targetPos);
				
				// IDEAL: Safe distance AND within weapon range
				if (actualDistance >= FixPoint(safeDistance) && actualDistance <= FixPoint(weaponRange)) {
					return candidatePos; // Perfect kiting position!
				}
			}
		}
	}
	
	// Phase 2: No safe tiles within range found - retreat further for safety
	for (int searchRadius = 0; searchRadius <= maxSearchRadius; searchRadius++) {
		for (int dx = -searchRadius; dx <= searchRadius; dx++) {
			for (int dy = -searchRadius; dy <= searchRadius; dy++) {
				// Only check tiles on the perimeter of current search radius
				if (searchRadius > 0 && abs(dx) != searchRadius && abs(dy) != searchRadius) {
					continue;
				}
				
				Coord candidatePos(idealRetreatPos.x + dx, idealRetreatPos.y + dy);
				
				// Check if tile exists and is passable
				if (!getMap().tileExists(candidatePos.x, candidatePos.y)) {
					continue;
				}
				
				if (!getMap().okayToPlaceStructure(candidatePos.x, candidatePos.y, 1, 1, false, nullptr)) {
					continue; // Tile not passable or occupied
				}
				
				// Calculate actual distance to target
				FixPoint actualDistance = blockDistance(candidatePos, targetPos);
				
				// FALLBACK: Just get to safe distance (even if out of range)
				if (actualDistance >= FixPoint(safeDistance)) {
					return candidatePos; // Safety first, can re-engage later
				}
			}
		}
	}
	
	// Phase 3: Emergency fallback - use original squad center retreat
	return squadRallyLocation;
}

Coord QuantBot::findBaseCentre(int houseID) {
	// First, calculate base center and validate it has a building
	int buildingCount = 0;
	int totalX = 0;
	int totalY = 0;

	std::vector<const StructureBase*> enemyBuildings;

	for (const StructureBase* pCurrentStructure : getStructureList()) {
		if (pCurrentStructure->getOwner()->getHouseID() == houseID && pCurrentStructure->getStructureSizeX() != 1) {
			// Collect buildings for center calculation and nearest building fallback
			enemyBuildings.push_back(pCurrentStructure);
			buildingCount++;
			totalX += pCurrentStructure->getX();
			totalY += pCurrentStructure->getY();
		}
	}

	if (buildingCount == 0) {
		return Coord::Invalid();
	}

	// Calculate mathematical center
	Coord calculatedCenter(totalX / buildingCount, totalY / buildingCount);

	// Check if the calculated center has a building on it
	for (const StructureBase* pStructure : enemyBuildings) {
		Coord structureLoc = pStructure->getLocation();
		Coord structureSize = pStructure->getStructureSize();
		
		// Check if calculated center falls within any building's footprint
		if (calculatedCenter.x >= structureLoc.x && 
			calculatedCenter.x < structureLoc.x + structureSize.x &&
			calculatedCenter.y >= structureLoc.y && 
			calculatedCenter.y < structureLoc.y + structureSize.y) {
			return calculatedCenter; // Center has a building, use it
		}
	}

	// If calculated center doesn't have a building, find nearest building to center
	const StructureBase* nearestBuilding = nullptr;
	FixPoint nearestDistance = FixPt_MAX;

	for (const StructureBase* pStructure : enemyBuildings) {
		FixPoint distance = blockDistance(calculatedCenter, pStructure->getLocation());
		if (distance < nearestDistance) {
			nearestDistance = distance;
			nearestBuilding = pStructure;
		}
	}

	// Simply return the nearest building to center
	if (nearestBuilding != nullptr) {
		return nearestBuilding->getLocation();
	}

	// Fallback to calculated center if all else fails
	return calculatedCenter;
}


Coord QuantBot::findSquadCenter(int houseID) {
	int squadSize = 0;

	int totalX = 0;
	int totalY = 0;

	for (const UnitBase* pCurrentUnit : getUnitList()) {
		if (pCurrentUnit && pCurrentUnit->getOwner()->getHouseID() == houseID
			&& pCurrentUnit->getItemID() != Unit_Carryall
			&& pCurrentUnit->getItemID() != Unit_Harvester
			&& pCurrentUnit->getItemID() != Unit_Frigate
			&& pCurrentUnit->getItemID() != Unit_MCV

			// Stop freeman making tanks roll forward
			&& !(currentGame && currentGame->techLevel > 6 && pCurrentUnit->getItemID() == Unit_Trooper)
			&& pCurrentUnit->getItemID() != Unit_Saboteur
			&& pCurrentUnit->getItemID() != Unit_Sandworm

			// Don't let troops moving to rally point contribute
			/*
			&& pCurrentUnit->getAttackMode() != RETREAT
			&& pCurrentUnit->getDestination().x != squadRallyLocation.x
			&& pCurrentUnit->getDestination().y != squadRallyLocation.y*/) {

			// Lets find the center of mass of our squad
			squadSize++;
			totalX += pCurrentUnit->getX();
			totalY += pCurrentUnit->getY();
		}

	}

	Coord squadCenterLocation = Coord::Invalid();

	if (squadSize > 0) {
		squadCenterLocation.x = totalX / squadSize;
		squadCenterLocation.y = totalY / squadSize;
	}

	return squadCenterLocation;
}

/**
	Set a rally / retreat location for all our military units.
	This should be near our base but within it
	The retreat mode causes all our military units to move
	to this squad rally location

*/
void QuantBot::retreatAllUnits() {

	// Set the new squad rally location
	squadRallyLocation = findSquadRallyLocation();
	squadRetreatLocation = findSquadRetreatLocation();

	// turning this off fow now
	//retreatTimer = MILLI2CYCLES(90000);

	// If no base exists yet, there is no retreat location
	if (squadRallyLocation.isValid() && squadRetreatLocation.isValid()) {
		for (const UnitBase* pUnit : getUnitList()) {
			if (pUnit && pUnit->getOwner() == getHouse()
				&& pUnit->getItemID() != Unit_Carryall
				&& pUnit->getItemID() != Unit_Sandworm
				&& pUnit->getItemID() != Unit_Harvester
				&& pUnit->getItemID() != Unit_MCV
				&& pUnit->getItemID() != Unit_Frigate) {

				doSetAttackMode(pUnit, RETREAT);
			}
		}
	}
}


/**
	In dune it is best to mass military units in one location.
	This function determines a squad leader by finding the unit with the most central location
	Amongst all of a players units.

	Rocket launchers and Ornithopters are excluded from having this role as on the
	battle field these units should always have other supporting units to work with

*/
    void QuantBot::checkAllUnits() {
        // Safety check: if our house is null (e.g., during game cleanup), don't check units
        if (getHouse() == nullptr) {
            return;
        }
        
        // Calculate current military value for ornithopter attack decisions
        int militaryValue = 0;
        if (currentGame) {
            for (Uint32 i = Unit_FirstID; i <= Unit_LastID; i++) {
                if (i != Unit_Carryall && i != Unit_Harvester && i != Unit_MCV && i != Unit_Sandworm) {
                    militaryValue += getHouse()->getNumItems(i) * currentGame->objectData.data[i][getHouse()->getHouseID()].price;
                }
            }
        }
        
        Coord squadCenterLocation = findSquadCenter(getHouse()->getHouseID());

        for (const UnitBase* pUnit : getUnitList()) {
            if (pUnit && pUnit->getOwner() == getHouse()) {
                switch (pUnit->getItemID()) {
                case Unit_MCV: {
                    const MCV* pMCV = static_cast<const MCV*>(pUnit);
                    if (pMCV != nullptr) {
                        //logDebug("MCV: forced: %d  moving: %d  canDeploy: %d",
                        //pMCV->wasForced(), pMCV->isMoving(), pMCV->canDeploy());

                        if (pMCV->canDeploy() && !pMCV->wasForced() && !pMCV->isMoving()) {
                            //logDebug("MCV: Deployed");
                            doDeploy(pMCV);
                        }
                        else if (!pMCV->isMoving() && !pMCV->wasForced()) {
                            Coord pos = findMcvPlaceLocation(pMCV);
                            doMove2Pos(pMCV, pos.x, pos.y, true);
                            /*
                            if(getHouse()->getNumItems(Unit_Carryall) > 0){
                                doRequestCarryallDrop(pMCV);
                            }*/
                        }
                    }
                } break;

                case Unit_Harvester: {
                    const Harvester* pHarvester = static_cast<const Harvester*>(pUnit);
                    if(pHarvester != nullptr && pHarvester->isActive()) {
                        // Existing check for early return with half spice
                        if(getHouse()->getCredits() < 1000 && pHarvester->getAmountOfSpice() >= HARVESTERMAXSPICE/2 
                            && getHouse()->getNumItems(Structure_HeavyFactory) == 0) {
                            doReturn(pHarvester);
                        }
                        
                        /* this needs to be fixed to make better, currently if they are trying to move somewhere it will trigger
                        // Check for idle harvesters
                        if(!pHarvester->isMoving() && !pHarvester->isHarvesting()) {
                            doSetAttackMode(pHarvester, GUARD);
                        }*/
                    }
                } break;

                case Unit_Carryall: {
                } break;

                case Unit_Frigate: {
                } break;

                case Unit_Sandworm: {
                } break;

                case Unit_Ornithopter: {
                    const UnitBase* pOrnithopter = pUnit;
                    
                    // We have enough ornithopters, they should attack enemy structures
                    if (!pOrnithopter->hasATarget() || !pOrnithopter->getTarget()->isVisible(getHouse()->getTeamID())) {
                        // Find closest enemy structure to squad rally point
                        Coord squadRallyPoint = findSquadRallyLocation();
                        const StructureBase* primaryTarget = nullptr;
                        FixPoint closestDistance = FixPt_MAX;

                        for (const StructureBase* pStructure : getStructureList()) {
                            if (pStructure->getOwner()->getTeamID() != getHouse()->getTeamID()) {
                                FixPoint distance = blockDistance(squadRallyPoint, pStructure->getLocation());
                                if (distance < closestDistance) {
                                    closestDistance = distance;
                                    primaryTarget = pStructure;
                                }
                            }
                        }

                        if (primaryTarget != nullptr) {
                            // Count rocket turrets belonging to the same player as our target
                            int targetPlayerRocketTurrets = 0;
                            for (const StructureBase* pStructure : getStructureList()) {
                                if (pStructure->getOwner()->getHouseID() == primaryTarget->getOwner()->getHouseID()
                                    && pStructure->getItemID() == Structure_RocketTurret) {
                                    targetPlayerRocketTurrets++;
                                }
                            }
                            
                            // Only attack if we have enough ornithopters for this specific player's defenses
                            int requiredOrnithopters = std::max(1, targetPlayerRocketTurrets);
                            
                            // Calculate current ornithopter percentage of total military units
                            int totalMilitaryUnits = 0;
                            int ornithopterCount = getHouse()->getNumItems(Unit_Ornithopter);
                            
                            // Count all military units (excluding harvesters, MCVs, carryalls)
                            for (Uint32 i = Unit_FirstID; i <= Unit_LastID; i++) {
                                if (i != Unit_Carryall && i != Unit_Harvester && i != Unit_MCV && i != Unit_Sandworm) {
                                    totalMilitaryUnits += getHouse()->getNumItems(i);
                                }
                            }
                            
                            // Calculate ornithopter percentage
                            FixPoint ornithopterPercentage = 0;
                            if (totalMilitaryUnits > 0) {
                                ornithopterPercentage = FixPoint(ornithopterCount) / FixPoint(totalMilitaryUnits);
                            }
                            
                            // New condition: attack if ornithopters > 20% of troops AND military value >= 40% of limit
                            bool hasEnoughOrnithoptersVsTurrets = (ornithopterCount >= requiredOrnithopters);
                            bool hasHighOrnithopterRatio = (ornithopterPercentage > 0.20_fix) && (militaryValue >= militaryValueLimit * 0.40_fix);
                            
                            if (hasEnoughOrnithoptersVsTurrets) {
                                logDebug("Ornithopter attack: sufficient vs turrets - ornis: %d >= required: %d", ornithopterCount, requiredOrnithopters);
                            } else if (hasHighOrnithopterRatio) {
                                logDebug("Ornithopter attack: high ratio condition - orni%%: %.1f%% (>20%%), mval: %d/%d (%.1f%% >= 40%%)", 
                                    ornithopterPercentage.toFloat() * 100.0f, militaryValue, militaryValueLimit, 
                                    (FixPoint(militaryValue) / FixPoint(militaryValueLimit)).toFloat() * 100.0f);
                            }
                            
                            if (hasEnoughOrnithoptersVsTurrets || hasHighOrnithopterRatio) {
                                // Check if any rocket turrets are in range of the nearest enemy building
                                const StructureBase* rocketTurretTarget = nullptr;
                                int rocketTurretRange = 0;
                                if (currentGame) {
                                    rocketTurretRange = currentGame->objectData.data[Structure_RocketTurret][getHouse()->getHouseID()].weaponrange;
                                }
                                
                                for (const StructureBase* pTurret : getStructureList()) {
                                    if (pTurret->getOwner()->getTeamID() != getHouse()->getTeamID() 
                                        && pTurret->getItemID() == Structure_RocketTurret) {
                                        FixPoint distanceToNearestBuilding = blockDistance(pTurret->getLocation(), primaryTarget->getLocation());
                                        if (distanceToNearestBuilding <= rocketTurretRange) {
                                            rocketTurretTarget = pTurret;
                                            break; // Prioritize first rocket turret found in range
                                        }
                                    }
                                }

                                // Choose target: prioritize rocket turrets in range, otherwise use nearest building
                                const StructureBase* finalTarget = rocketTurretTarget ? rocketTurretTarget : primaryTarget;
                                doAttackObject(pOrnithopter, finalTarget, true);
                            } else {
                                // Not enough ornithopters to attack this specific player - patrol defensively
                                if (!pOrnithopter->hasATarget() && !pOrnithopter->wasForced()) {
                                    Coord ownBaseCentre = findBaseCentre(getHouse()->getHouseID());
                                    if (ownBaseCentre.isValid() && ownBaseCentre != pOrnithopter->getGuardPoint()) {
                                        const_cast<UnitBase*>(pOrnithopter)->setGuardPoint(ownBaseCentre.x, ownBaseCentre.y);
                                    }
                                }
                            }
                        } else {
                            // No enemy structures found at all - patrol defensively
                            if (!pOrnithopter->hasATarget() && !pOrnithopter->wasForced()) {
                                Coord ownBaseCentre = findBaseCentre(getHouse()->getHouseID());
                                if (ownBaseCentre.isValid() && ownBaseCentre != pOrnithopter->getGuardPoint()) {
                                    const_cast<UnitBase*>(pOrnithopter)->setGuardPoint(ownBaseCentre.x, ownBaseCentre.y);
                                }
                            }
						}
					}
				// Note: Defensive behavior (patrolling own base) is handled above in the various else clauses
			} break;

                default: {
                    int squadRadius = lround(FixPoint::sqrt(getHouse()->getNumUnits()
                        - getHouse()->getNumItems(Unit_Harvester)
                        - getHouse()->getNumItems(Unit_Carryall)
                        - getHouse()->getNumItems(Unit_Ornithopter)
                        - getHouse()->getNumItems(Unit_Sandworm)
                        - getHouse()->getNumItems(Unit_MCV))) + 1;

                    if (pUnit->getOwner()->getHouseID() != pUnit->getOriginalHouseID()) {
                        // If its a devastator and its not ours, blow it up!!
                        if (pUnit->getItemID() == Unit_Devastator) {
                            const Devastator* pDevastator = static_cast<const Devastator*>(pUnit);
                            doStartDevastate(pDevastator);
                            doSetAttackMode(pDevastator, HUNT);
                        }
                        /*
                        else if (pUnit->getItemID() == Unit_Ornithopter) {
                            if (pUnit->getAttackMode() != HUNT) {
                                doSetAttackMode(pUnit, HUNT);
                            }
                        }*/
                        else if (pUnit->getItemID() == Unit_Harvester) {
                            const Harvester* pHarvester = static_cast<const Harvester*>(pUnit);
                            if (pHarvester->getAmountOfSpice() >= HARVESTERMAXSPICE / 5) {
                                doReturn(pHarvester);
                            }
                            else {
                                doMove2Pos(pUnit, squadCenterLocation.x, squadCenterLocation.y, true);
                            }
                        }
                        else {
                            // Send deviated unit to squad centre
                            if (pUnit->getAttackMode() != AREAGUARD) {
                                doSetAttackMode(pUnit, AREAGUARD);
                            }

                            if (blockDistance(pUnit->getLocation(), squadCenterLocation) > squadRadius - 1) {
                                doMove2Pos(pUnit, squadCenterLocation.x, squadCenterLocation.y, true);
                            }
                        }
                    }
                    else if ((pUnit->getItemID() == Unit_Launcher || pUnit->getItemID() == Unit_Deviator)
                        && pUnit->hasATarget() && (difficulty != Difficulty::Easy)) {
                        // Special logic to keep launchers away from harm
                        const ObjectBase* pTarget = pUnit->getTarget();
                        if (pTarget != nullptr) {
                            if (blockDistance(pUnit->getLocation(), pTarget->getLocation()) <= 6 && pTarget->getItemID() != Unit_Ornithopter) {
                                // Tactical retreat: move to optimal firing distance instead of squad center
                                Coord tacticalRetreatPos = findTacticalRetreatLocation(pUnit, pTarget);
                                
                                if (tacticalRetreatPos.isValid()) {
                                    doSetAttackMode(pUnit, AREAGUARD); // Change mode to stop launchers freezing
                                    doMove2Pos(pUnit, tacticalRetreatPos.x, tacticalRetreatPos.y, true);
                                    logDebug("Launcher tactical retreat from close target: moving to (%d,%d)", tacticalRetreatPos.x, tacticalRetreatPos.y);
                                } else {
                                    // Fallback to original behavior
                                    doSetAttackMode(pUnit, AREAGUARD); // Change mode to stop launchers freezing
                                    doMove2Pos(pUnit, squadCenterLocation.x, squadCenterLocation.y, true);
                                }
                            }
                        }
                    }
                    else if (pUnit->getItemID() != Unit_Ornithopter && pUnit->getAttackMode() != HUNT && !pUnit->hasATarget() && !pUnit->wasForced()) {
                        if (pUnit->getAttackMode() == AREAGUARD && squadCenterLocation.isValid() && (gameMode != GameMode::Campaign)) {
                            if (blockDistance(pUnit->getLocation(), squadCenterLocation) > squadRadius) {
                                if (!pUnit->hasATarget()) {
                                    doMove2Pos(pUnit, squadCenterLocation.x, squadCenterLocation.y, false);
                                }
                            }
                        }
                        else if (pUnit->getAttackMode() == RETREAT) {
                            if (blockDistance(pUnit->getLocation(), squadRetreatLocation) > squadRadius + 2 && !pUnit->wasForced()) {
                                if (pUnit->getHealth() < pUnit->getMaxHealth()) {
                                    doRepair(pUnit);
                                }
                                doMove2Pos(pUnit, squadRetreatLocation.x, squadRetreatLocation.y, true);
                            }
                            else {
                                // We have finished retreating back to the rally point
                                doSetAttackMode(pUnit, AREAGUARD);
                            }
                        }
                        else if (pUnit->getAttackMode() == GUARD
                            && ((pUnit->getDestination() != squadRallyLocation) || (blockDistance(pUnit->getLocation(), squadRallyLocation) <= squadRadius))) {
                            // A newly deployed unit has reached the rally point, or has been diverted => Change it to area guard
                            doSetAttackMode(pUnit, AREAGUARD);
                        }
                    }
                } break;
            }
        }
    }
}
