# Dune Legacy 0.98.5 vs 0.98.4 - Business Logic Changes Documentation

## Overview
This document outlines all business logic changes between Dune Legacy release-0.98.4 and 0.98.5 branches, providing detailed information for reimplementation. The changes span across multiple areas including AI improvements, performance optimizations, bug fixes, and new features.

## Summary of Changes
- **25 commits** between release-0.98.4 and origin/0.98.5
- **Major focus areas**: QuantBot AI improvements, performance optimizations, segfault fixes, FPS unlocking
- **Version update**: 0.98.4 → 0.98.5

---

## 1. Performance & Display Changes

### 1.1 FPS Unlocking (Commit: b0b113e)
**Impact**: Major performance improvement allowing unlimited FPS

**Changes**:
- **New VSync Option**: Added `vSync` setting to video configuration
  - Default: `false` (VSync disabled)
  - Configurable in Options Menu
  - Controls SDL_HINT_RENDER_VSYNC setting

- **Frame Limiting Changes**:
  - Default `frameLimit` changed from `true` to `false`
  - Removed hard-coded 32ms frame delays (was capping at ~31 FPS)
  - Replaced with minimal 1ms delays to yield CPU time

- **Scroll Speed Improvements**:
  - Added frame-time-aware scrolling calculations
  - Scroll speed now scales properly with actual frame rate
  - Formula: `pixelsPerSecond = scrollSpeed * (1000.0f / 16.67f) * frameTimeMs / 1000.0f`

**Files Modified**:
- `include/DataTypes.h`: Added `vSync` field to video settings
- `include/Menu/OptionsMenu.h`: Added vSync checkbox
- `include/ScreenBorder.h`: Updated scroll methods with frame time parameters
- `src/ScreenBorder.cpp`: Implemented frame-time-aware scrolling
- `src/Menu/OptionsMenu.cpp`: VSync UI controls and configuration saving
- `src/main.cpp`: Default settings and SDL hint configuration
- `src/Menu/MenuBase.cpp`: Removed frame limiting
- `src/MapEditor/MapEditor.cpp`: Removed frame limiting

---

## 2. Nuclear Missile Targeting Fix (Commit: 7cd6e49)

**Impact**: Critical AI behavior fix for Harkonnen/Sardaukar nuclear missiles

**Problem**: Nuclear missile targeting was commented out, making AI ineffective
**Solution**: Re-enabled proper targeting logic

**Changes**:
- **Palace.cpp**: Restored nuclear missile targeting for AI players
  - Harkonnen/Sardaukar now target closest enemy structure with nuclear missiles
  - Uses `findClosestTargetStructure()` and `doLaunchDeathhand()`

**Business Logic**:
```cpp
// Before: Nuclear missiles were never launched by AI
/* 
const StructureBase* closestStructure = findClosestTargetStructure();
if(closestStructure) {
    Coord temp = closestStructure->getClosestPoint(getLocation());
    doLaunchDeathhand(temp.x, temp.y);
}
*/

// After: AI properly launches nuclear missiles
const StructureBase* closestStructure = findClosestTargetStructure();
if(closestStructure) {
    Coord temp = closestStructure->getClosestPoint(getLocation());
    doLaunchDeathhand(temp.x, temp.y);
}
```

---

## 3. AI Helper System (Commit: 855f1dd)

**Impact**: New feature allowing AI assistance in single-player skirmish

**New Features**:
- **AI Helper Selection**: Dropdown in Single Player Skirmish menu
- **Multiple Players Per House**: Enabled when AI helper is selected
- **AI Helper Options**:
  - None (default)
  - QuantBot: Easy, Medium, Hard, Brutal
  - AI Player: Easy, Medium, Hard

**Implementation**:
- `SinglePlayerSkirmishMenu.h`: Added AI helper UI components
- `SinglePlayerSkirmishMenu.cpp`: 
  - AI helper dropdown with 7 options
  - Automatic assignment of AI helper to human player's house
  - Maps selection to AI class names (`qBotVeryEasy`, `qBotMedium`, etc.)

**Business Logic**:
- When AI helper selected: `setMultiplePlayersPerHouse(true)`
- AI helper added as second player to human player's house
- Both human and AI helper control same house units/structures

---

## 4. QuantBot AI Improvements

### 4.1 Major QuantBot Overhaul (Commit: 548d4a1)
**Impact**: Comprehensive AI behavior improvements

**Key Changes**:
- **438 insertions, 141 deletions** in QuantBot.cpp
- Enhanced tactical decision making
- Improved unit management and coordination
- Better resource allocation algorithms

### 4.2 Harvester Limit Increases (Commit: 1630f04)
**Impact**: Significantly improved AI economy on larger maps

**Changes by Difficulty**:
- **Brutal Difficulty**:
  - 32x32 maps: 5 → 10 harvesters (+100%)
  - 64x64 maps: 10 → 20 harvesters (+100%)  
  - 128x128 maps: 70 → 100 harvesters (+43%)
  - Military value limits also increased proportionally

- **Spice Threshold**: Changed from 1500 to 2000 spice per harvester
  - Formula: `maxHarvesters = totalSpice / 2000` (was 1500)

### 4.3 Unit Composition Rebalancing (Commit: 4955e56)
**Impact**: More balanced and effective AI armies

**Harkonnen Changes**:
- Launcher percentage: 70% → 50% (reduced dominance)
- Special percentage: 10% → 15% (more variety)
- Siege percentage: 10% → 15% (more siege units)
- Tank percentage: 0% → 20% (added tank production)

**Atreides Changes**:
- Launcher percentage: 20% → 25% (slight increase)
- Special percentage: 65% → 50% (reduced dominance)
- Siege percentage: 0% → 10% (added siege capability)
- Tank percentage: 0% → 10% (added tank production)
- Ornithopter percentage: 15% → 5% (reduced air focus)

### 4.4 Kiting System Improvements (Commits: bb8d7e6, 991d02a)
**Impact**: Better tactical unit positioning and retreat behavior

**New Features**:
- **Launcher Retreat Timer**: Added `launcherRetreatTimer` field
- **Improved Retreat Logic**: Moved from damage handler to unit checking
- **Tactical Positioning**: Better positioning for ranged units

**Key Changes**:
- Removed spam-prone retreat logic from `onDamage()`
- Added cooldown system for launcher retreats
- Better ownership checking before commanding units
- Health threshold for repairs: 60% → 70%

### 4.5 Build Optimizations (Commit: 6cab3db)
**Impact**: More efficient AI construction decisions
- Optimized build priority algorithms
- Better resource allocation for construction
- Improved timing for structure placement

---

## 5. Segfault Fixes and Stability

### 5.1 Damage Tracking Fix (Commit: 2b9e90c)
**Impact**: Critical stability fix for damage tracking system

**Problem**: Segfaults in damage tracking when objects were destroyed
**Solution**: Moved damage tracking from ObjectBase to UnitBase

**Changes**:
- **ObjectBase.cpp**: Removed damage tracking code (13 lines deleted)
- **UnitBase.cpp**: Added proper null checking and damage tracking
- **Logic**: Only track damage for valid objects with proper ownership

### 5.2 Null Pointer Fixes (Commit: 5ce4164)
**Impact**: Additional stability improvements
- Added null pointer checks in QuantBot
- Fixed potential crashes in Palace operations
- Improved error handling throughout AI systems

### 5.3 Object Pointer Safety (Commits: d8c1d1e, 0517986, ffb3ab5)
**Impact**: Multiple segfault prevention measures
- Enhanced null checking in ObjectPointer operations
- Improved object lifecycle management
- Added defensive programming patterns

---

## 6. Minor Changes and Fixes

### 6.1 Integer Conversion Fixes (Commits: f189bed, 8fad181)
- Fixed integer precision warnings in QuantBot
- Improved type safety in calculations

### 6.2 Website Updates (Commit: 29a6c16)
- Updated news section with 33 new lines
- Prepared for 0.98.5 release announcement

### 6.3 Windows Compatibility (Commit: fcfea77)
- Added .gitignore entries for Windows builds
- Improved cross-platform compatibility

---

## 7. Configuration Changes

### 7.1 Version Updates (Commit: 7cd6e49)
Updated version numbers across all build systems:
- `CMakeLists.txt`: Version 0.98.4 → 0.98.5
- `ChangeLog`: Added 0.98.5 release notes
- All installer scripts (NSIS, DMG): Updated version strings
- Xcode project: Version and build number updates
- `config.h`: Version macro updates

---

## 8. Implementation Guide

### 8.1 Critical Changes for Reimplementation

1. **FPS Unlocking**: 
   - Add VSync option to video settings
   - Remove hard-coded frame delays
   - Implement frame-time-aware scrolling

2. **Nuclear Missile Fix**:
   - Ensure AI nuclear missile targeting is enabled
   - Verify `findClosestTargetStructure()` functionality

3. **QuantBot Improvements**:
   - Update harvester limits for all difficulty levels
   - Implement new unit composition ratios
   - Add launcher retreat timer system

4. **AI Helper System**:
   - Add dropdown to skirmish menu
   - Implement multiple players per house
   - Map AI helper selections to class names

5. **Stability Fixes**:
   - Move damage tracking from ObjectBase to UnitBase
   - Add comprehensive null pointer checking
   - Implement defensive programming patterns

### 8.2 Testing Priorities

1. **Performance**: Verify FPS unlocking works correctly
2. **AI Behavior**: Test nuclear missile targeting
3. **Stability**: Ensure no segfaults in damage tracking
4. **Balance**: Verify unit composition changes
5. **UI**: Test AI helper dropdown functionality

### 8.3 Configuration Updates Required

1. Update all version strings to 0.98.5
2. Set default `frameLimit = false`
3. Add `vSync = false` to default config
4. Update installer scripts and build configurations

---

## 9. Risk Assessment

### 9.1 High Risk Changes
- **FPS Unlocking**: May affect game balance and timing
- **Damage Tracking**: Critical for AI decision making
- **Nuclear Missile**: Significant AI behavior change

### 9.2 Medium Risk Changes
- **Unit Composition**: May affect game balance
- **Harvester Limits**: Economic impact on gameplay
- **AI Helper System**: New feature complexity

### 9.3 Low Risk Changes
- **Version Updates**: Cosmetic changes
- **Website Updates**: Documentation only
- **Build Fixes**: Development environment improvements

---

## 10. Conclusion

The 0.98.5 release represents a significant improvement over 0.98.4, with major performance enhancements, AI improvements, and stability fixes. The FPS unlocking feature alone provides a substantial user experience improvement, while the QuantBot enhancements make AI opponents more challenging and engaging.

Key areas requiring careful attention during reimplementation:
1. Performance improvements (FPS unlocking)
2. AI behavior changes (nuclear missiles, unit composition)
3. Stability fixes (segfault prevention)
4. New features (AI helper system)

The changes maintain backward compatibility while providing substantial improvements to gameplay, performance, and stability.

